<?php

namespace Inside\Notify\Services;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Notify\Messages\SmsMessage;
use Inside\Notify\Notifications\InsideNotification;
use Inside\Support\Str;
use libphonenumber\NumberParseException;
use Twilio\Exceptions\ConfigurationException;
use Twilio\Exceptions\TwilioException;
use Twilio\Rest\Client;
use Twilio\Rest\Pricing\V2\CountryInstance;

class TwilioService extends SmsDriverService implements SmsDriverServiceInterface
{
    protected ?Client $smsTransactionApi = null;

    public function isAvailable(): bool
    {
        return ! is_null($this->config['account_sid']) &&
            ! is_null($this->config['auth_token']);
    }

    /**
     * Load and prepare sms transaction api
     * @throws ConfigurationException
     */
    protected function getSmsTransactionApi(): Client
    {
        if (is_null($this->smsTransactionApi)) {
            $this->smsTransactionApi = new Client(
                $this->config['account_sid'],
                $this->config['auth_token']
            );

            if (config('app.debug', false)) {
                $this->smsTransactionApi->setLogLevel('debug');
            }
        }

        return $this->smsTransactionApi;
    }

    public function overQuota(): bool
    {
        // First check our instance quota
        if (! is_null($this->getQuota()) &&
            $this->getSmsCountForCurrentMonth() > $this->getQuota()) {
            return true;
        }

        // Check from sendinblue credits
        if ($this->getRemainingCreditsForCurrentPlan() <= 0.0) {
            return true;
        }

        return false;
    }

    /**
     * @param  mixed  $message
     * @return bool
     * @throws ConfigurationException
     * @throws TwilioException
     * @throws Exception
     */
    public function send($message): bool
    {
        if (! $message instanceof SmsMessage) {
            throw new Exception(
                '[TwilioService::send] message is not a SmsMessage => ['.get_class($message).']'
            );
        }
        if ($this->overQuota()) {
            Log::warning('[TwilioService::send] quota exceeded');

            return false;
        }
        $result = $this->getSmsTransactionApi()->messages->create(
            $message->getRecipient(),
            [
                'body' => $message->getContent(),
                'from' => $message->getSender(),
                'smartEncoded' => true,
            ]
        );
        if ($result->errorCode && ! empty($result->errorMessage)) {
            Log::error('[TwilioService::send] failed to send sms <'.$result->errorMessage.'>');

            return false;
        }
        Log::debug('[TwilioService::send] '.json_encode($result->toArray()));
        $this->addSmsCount((int) $result->numSegments);

        return true;
    }

    /**
     * @return float|null
     * @throws ConfigurationException
     * @throws TwilioException
     */
    public function getPricePerSMS(): ?float
    {
        /** @var CountryInstance $frenchPrices */
        $frenchPrices = $this->getSmsTransactionApi()
            ->pricing
            ->v1
            ->messaging
            ->countries('fr')
            ->fetch();
        foreach ($frenchPrices->inboundSmsPrices as $price) {
            if ($price['number_type'] === 'mobile') {
                return (float) $price['current_price'];
            }
        }

        return null;
    }

    /**
     * @param  string  $number
     * @return int|null
     */
    public function mobileNumberVerificationSend(string $number): ?int
    {
        try {
            $verification = $this->getSmsTransactionApi()
                ->verify
                ->v2
                ->services($this->config['verification_sid'])
                ->verifications
                ->create($number, 'sms');
        } catch (TwilioException $exception) {
            Log::error('[TwilioService::mobileNumberVerificationSend] '.$exception->getMessage());

            return null;
        }

        return intval($verification->sid);
    }

    /**
     * @param  string  $number
     * @param  string  $code
     * @return bool
     */
    public function mobileNumberVerificationCheck(string $number, string $code): bool
    {
        try {
            $verificationCheck = $this->getSmsTransactionApi()
                ->verify
                ->v2
                ->services($this->config['verification_sid'])
                ->verificationChecks
                ->create(['to' => $number, 'code' => $code]);
        } catch (TwilioException $exception) {
            Log::error('[TwilioService::mobileNumberVerificationCheck] '.$exception->getMessage());

            return false;
        }

        return $verificationCheck->status === 'approved';
    }

    /**
     * @param  InsideNotification  $notification
     * @return SmsMessage
     * @throws NumberParseException
     * @throws Exception
     */
    public function getMessageFromNotification(InsideNotification $notification): SmsMessage
    {
        $user = $notification->getUser();
        if (is_null($user)) {
            throw new Exception('[TwilioService::getMessageFromNotification] needs a user recipient');
        }
        $locale = $user->langcode;
        if (is_string($locale) && in_array($locale, list_languages())) {
            Lang::setLocale($locale);
        }

        // Prepare en format data
        $data = $notification->getData() ?? [];
        $transformedData = [];

        $model = $notification->getModel();
        if (isset($notification->getNotificationType()->data['fields']) && ! is_null($model)) {
            $transformedData = App::make(Transformer::class)->transform(
                $model,
                $notification->getNotificationType()->data['fields']
            );
        }

        $data = Arr::dot(array_merge($data, $transformedData));
        $data = array_map(
            function ($item) {
                return $item === [] ? '' : trim($item);
            },
            $data
        ); // Remove \n at end

        $data['site'] = Str::ascii(config('app.name'));
        $data['link'] = $notification->getContentUrl();
        $data['link'] = isset($data['link']) ? url($data['link']) : null;

        // Prepare content
        $content = $notification->getNotificationType()->data['sms']['content'] ??
            ($notification->getData()['extra']['sms']['content'] ?? '');
        if (! is_null($model)) {
            $content = $notification->getTranslationKey(
                $content,
                $model,
                $notification->getData() ?? []
            );
            $content = Lang::get($content, $data, $locale);

            $content = $notification->replaceLastVariables($content, $model);
        }
        $mobile = $user->{$this->mobileFieldName ?? 'mobile'};
        if (is_null($mobile)) {
            throw new Exception(
                '[TwilioService::getMessageFromNotification] failed to get mobile from user <'.
                $user->uuid.'>'
            );
        }

        return (new SmsMessage())
            ->to($mobile)
            ->from($this->sender ?? config('name.from.name'))
            ->content($content);
    }

    /**
     * get remaining credit on twilio
     */
    public function getRemainingCreditsForCurrentPlan(): float
    {
        try {
            $balance = $this->getSmsTransactionApi()->balance->fetch();
        } catch (Exception $exception) {
            return 0.0;
        }
        if (is_null($this->currency)) {
            $this->currency = $balance->currency;
        }

        return (float) $balance->balance;
    }
}
