<?php

declare(strict_types=1);

namespace Inside\Notify\Services;

use Exception;
use GuzzleHttp\Client as GuzzleClient;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Notify\Messages\SendinBlueSmsMessage;
use Inside\Notify\Messages\SmsMessage;
use Inside\Notify\Models\SendinbluePlan;
use Inside\Notify\Models\SmsQuota;
use Inside\Notify\Notifications\InsideNotification;
use SendinBlue\Client\Api\AccountApi;
use SendinBlue\Client\Api\TransactionalSMSApi;
use SendinBlue\Client\ApiException;
use SendinBlue\Client\Configuration;
use SendinBlue\Client\Model\GetAccountPlan;
use SendinBlue\Client\Model\SendSms;
use SendinBlue\Client\Model\SendTransacSms;

/**
 * Sendinblue Service
 */
class SendinBlueService extends SmsDriverService implements SmsDriverServiceInterface
{
    /**
     * @var TransactionalSMSApi|null
     */
    protected $smsTransactionApi;

    /**
     * @var AccountApi|null
     */
    protected $accountApi;

    /**
     * is Sendinblue service available ?
     *
     * @return bool
     */
    public function isAvailable(): bool
    {
        return $this->config['api_key'] !== null;
    }

    /**
     * Did we spend all allowed sms for this plan ? ( from config )
     * Note: his is independent of subscription plan
     *
     * @return bool
     */
    public function overQuota(): bool
    {
        // First check our instance quota
        if ($this->getSmsCountForCurrentMonth() > $this->getQuota()) {
            return true;
        }

        // Check from sendinblue credits
        if (is_null($this->getRemainingCreditsForCurrentPlan()) ||
            $this->getUsedCreditsForCurrentPlan() > $this->getRemainingCreditsForCurrentPlan()) {
            return true;
        }

        return false;
    }

    /**
     * prepare SendinbluePlan depending on current sms plan
     *
     * @return SendinbluePlan|null
     */
    public function prepareSendinblueWithCurrentPlan(): ?SendinbluePlan
    {
        try {
            /** @var GetAccountPlan $smsPlan */
            $smsPlan = $this->getCurrentPlanFromApi();
            if (! is_null($smsPlan->getStartDate()) && ! is_null($smsPlan->getEndDate())) {
                return SendinbluePlan::create([
                    'remaining_credits' => $smsPlan->getCredits(),
                    'start_date' => $smsPlan->getStartDate(),
                    'end_date' => $smsPlan->getEndDate(),
                ]);
            }
        } catch (Exception $exception) {
            Log::warning(
                '[SendinBlueService::prepareSendinblueWithCurrentPlan] we did not find any sms plan in sendinBlue'
            );
        }

        return null;
    }

    /**
     * get updated sms plan from sendinblue current plan
     *
     * @return SendinbluePlan|null
     */
    public function getSendinblueFromCurrentPlan(): ?SendinbluePlan
    {
        $plan = $this->getCurrentPlan();
        if (is_null($plan)) {
            return $this->prepareSendinblueWithCurrentPlan();
        }

        try {
            /** @var GetAccountPlan $smsPlan */
            $smsPlan = $this->getCurrentPlanFromApi();
            if (! is_null($smsPlan->getStartDate()) && ! is_null($smsPlan->getEndDate())) {
                $plan->update([
                    'remaining_credits' => $smsPlan->getCredits(),
                ]);

                return $plan;
            }
        } catch (Exception $exception) {
            Log::warning(
                '[SendinBlueService::syncSendinblueFromCurrentPlan] we did not find any sms plan in sendinblue'
            );
        }

        return null;
    }

    /**
     * Get Currently plan
     *
     * @return float
     */
    public function getUsedCreditsForCurrentPlan(): float
    {
        $plan = $this->getSendinblueFromCurrentPlan();
        if (is_null($plan)) {
            return 0.0; // No plan set yet
        }

        return $plan->used_credits;
    }

    /**
     * @return float
     */
    public function getRemainingCreditsForCurrentPlan(): ?float
    {
        $plan = $this->getSendinblueFromCurrentPlan();
        if (is_null($plan)) {
            return null; // No plan set yet
        }

        return $plan->remaining_credits;
    }

    /**
     * @param  SendinBlueSmsMessage  $message
     * @return bool
     * @throws Exception
     */
    public function send($message): bool
    {
        if (! $message instanceof SendinBlueSmsMessage) {
            throw new Exception(
                '[SendinBlueService::send] message is not a SendinBlueSmsMessage => ['.get_class($message).']'
            );
        }
        if ($this->overQuota()) {
            Log::warning('[SendinBlueService::send] quota exceeded');

            return false;
        }
        $sendTransactionSms = new SendTransacSms($message->toArray());
        try {
            $this->updateQuota($this->getSmsTransactionApi()->sendTransacSms($sendTransactionSms));
        } catch (Exception $e) {
            Log::error('[SendinBlueService::send] failed to send sms <'.$e->getMessage().'>');

            return false;
        }

        return true;
    }

    /**
     * @throws \libphonenumber\NumberParseException
     * @throws Exception
     */
    public function getMessageFromNotification(InsideNotification $notification): SendinBlueSmsMessage
    {
        $user = $notification->getUser();
        if (is_null($user)) {
            throw new Exception('[SendingBlueService::getMessageFromNotification] needs a user recipient');
        }
        $locale = $user->langcode;
        if (is_string($locale) && in_array($locale, list_languages())) {
            Lang::setLocale($locale);
        }

        // Prepare en format data
        $data = $notification->getData() ?? [];
        $transformedData = [];

        if (isset($notification->getNotificationType()->data['fields'])) {
            $transformer = new ContentTransformer();
            $transformedData = $transformer->transform(
                $notification->getModel(),
                $notification->getNotificationType()->data['fields']
            );
        }
        $data = Arr::dot(array_merge($data, (array) $transformedData));
        $data = array_map(
            function ($item) {
                return $item === [] ? '' : trim($item);
            },
            $data
        ); // Remove \n at end

        // Prepare content
        $content = $notification->getNotificationType()->data['sms']['content'] ??
            ($notification->getData()['extra']['sms']['content'] ?? '');
        if (! is_null($notification->getModel())) {
            $content = $notification->getTranslationKey(
                $content,
                $notification->getModel(),
                $notification->getData() ?? []
            );
            $content = Lang::get($content, $data, $locale);
            $content = Lang::get($content, $data, $locale);

            $content = $notification->replaceLastVariables($content, $notification->getModel());
        }
        $mobile = $user->{$this->mobileFieldName ?? 'mobile'};
        if (is_null($mobile)) {
            throw new Exception(
                '[SendinblueService::getMessageFromNotification] failed to get mobile from user <'.
                $user->uuid.'>'
            );
        }

        return (new SendinBlueSmsMessage())
            ->to($mobile)
            ->from($this->sender ?? config('name.from.name'))
            ->content($content);
    }

    /**
     * @return float
     * @throws ApiException
     */
    public function getAccountRemainSmsCredits(): float
    {
        $result = $this->getAccountApi()->getAccount();
        /** @var GetAccountPlan $plan */
        foreach ($result['plan'] as $plan) { // @phpstan-ignore-line
            if ($plan->getType() != GetAccountPlan::TYPE_SMS) {
                continue;
            }

            return $plan->getCredits();
        }

        return 0.0;
    }

    /**
     * Get current sms plan
     *
     * @return GetAccountPlan|null
     * @throws ApiException
     */
    public function getCurrentPlanFromApi(): ?GetAccountPlan
    {
        $result = $this->getAccountApi()->getAccount();
        /** @var GetAccountPlan $plan */
        foreach ($result['plan'] as $plan) { // @phpstan-ignore-line
            if ($plan->getType() != GetAccountPlan::TYPE_SMS) {
                continue;
            }

            return $plan;
        }

        return null;
    }

    /**
     * Update quota
     *
     * @param  SendSms  $sendSms
     */
    protected function updateQuota(SendSms $sendSms): void
    {
        Log::debug('[SendinBlueService::sent] '.$sendSms);
        $quota = $this->getQuotaForCurrentMonth();
        if (is_null($quota)) {
            $quota = SmsQuota::create([
                'sms_count' => $sendSms->getSmsCount(),
            ]);
        }
        $quota->increment($sendSms->getSmsCount());

        /** @var SendinbluePlan $plan */
        $plan = $this->getCurrentPlan();
        $plan->update([
            'used_credits' => $sendSms->getUsedCredits(),
            'remaining_credits' => $sendSms->getRemainingCredits(),
        ]);
    }

    /**
     * Get Sendinblue API configuration
     *
     * @return Configuration
     */
    protected function getConfig(): Configuration
    {
        return Configuration::getDefaultConfiguration()->setApiKey('api-key', $this->config['api_key']);
    }

    /**
     * Load and prepare sms transaction api
     *
     * @return TransactionalSMSApi
     */
    protected function getSmsTransactionApi(): TransactionalSMSApi
    {
        if (is_null($this->smsTransactionApi)) {
            $this->smsTransactionApi = new TransactionalSMSApi(
                new GuzzleClient(),
                $this->getConfig()
            );
        }

        return $this->smsTransactionApi;
    }

    /**
     * Get Account API
     *
     * @return AccountApi
     */
    protected function getAccountApi(): AccountApi
    {
        if (is_null($this->accountApi)) {
            $this->accountApi = new AccountApi(
                new GuzzleClient(),
                $this->getConfig()
            );
        }

        return $this->accountApi;
    }

    /**
     * get Sendinblue Sms current plan
     *
     * @return SendinbluePlan|null
     */
    protected function getCurrentPlan(): ?SendinbluePlan
    {
        return SendinbluePlan::whereDate('start_date', '>=', now()->toDateString())
            ->whereDate('end_date', '<=', now()->toDateString())->first();
    }
}
