<?php

declare(strict_types=1);

namespace Inside\Notify\Services;

use Carbon\Carbon;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Services\Queries\ContentQueryHelper;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Database\Eloquent\Builder;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationRole;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Models\NotificationUserView;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role as RoleService;
use Inside\Slug\Facades\ContentUrlGenerator;

/**$notificationType, $subscriber, $content, ['from' => $from->uuid]switch(
 * Notification service.
 *
 * @category Class
 * @package  Inside\Notify\Services\NotificationService
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
final class NotificationService
{
    use HasSpecialLanguageKeys;

    /**
     * Transform notification collection or pagination
     *
     * @param mixed $notifications
     *
     * @return array
     */
    public function transform($notifications = null): array
    {
        if ($notifications == null) {
            return [];
        }

        $transformed = [];
        $content = null;

        foreach ($notifications as $item) {
            $user = Auth::user();
            $views = ! is_null($user) ? $item->getUserView($user->uuid) : null;
            $data = $item->toArray();
            $from = null;
            $transformedData = [];
            $transformedExtraData = [];

            $data['type'] = $item->trigger()->pluck('data')->first();
            $data['view_at'] = $views->view_at ?? null;
            $data['read_at'] = $views->read_at ?? null;

            $extra = $data['data']['extra'] ?? null;

            // Try to translate
            $locale = $item->notifiable_langcode;
            $defaultLocale = Arr::first(list_languages());
            if ($item->notifiable_langcode === '--') {
                $locale = $defaultLocale;
            }
            if (! is_null($user)) {
                $user = Users::find($user->uuid);
                if ($user) {
                    $locale = $user->langcode;
                } else {
                    $locale = $defaultLocale;
                }
            }

            if (isset($data['data']['from'])) {
                if (is_string($data['data']['from'])) {
                    $from = Users::find($data['data']['from']);
                }
                if ($from instanceof Users) {
                    $data['author'] = $from->status || config('users.decommissioning', false) !== true
                        ? [
                            'uuid' => $from->uuid,
                            'lastname' => $from->lastname,
                            'firstname' => $from->firstname,
                            'fullname' => trim($from->firstname.' '.$from->lastname),
                            'image' => protected_file_url($from, 'image', true, 'avatar'),
                        ]
                        : [
                            'uuid' => $from->uuid,
                            'lastname' => Lang::getFromJSON('anonymous.user.lastname'),
                            'firstname' => Lang::getFromJSON('anonymous.user.firstname'),
                            'fullname' => trim(
                                Lang::getFromJSON('anonymous.user.firstname').' '.Lang::getFromJSON(
                                    'anonymous.user.lastname'
                                )
                            ),
                            'image' => null,
                        ];
                }
            } else {
                $data['author'] = [
                    'lastname' => Lang::getFromJSON('anonymous.user.lastname'),
                    'firstname' => Lang::getFromJSON('anonymous.user.firstname'),
                    'fullname' => trim(
                        Lang::getFromJSON('anonymous.user.firstname').' '.Lang::getFromJSON(
                            'anonymous.user.lastname'
                        )
                    ),
                ];
            }

            if ($data['notifiable_type'] && $data['notifiable_uuid']) {
                if ($data['notifiable_type'] == 'Inside\Authentication\Models\User') {
                    $data['notifiable_type'] = 'Inside\Content\Models\Contents\Users';
                }

                if ($item->type == 'group' || $item->type == 'workflow') {
                    $query = call_user_func($data['notifiable_type'].'::withoutGlobalScopes');
                    $content = $query->where('uuid', $data['notifiable_uuid'])->first();
                } else {
                    $content = call_user_func($data['notifiable_type'].'::find', $data['notifiable_uuid']);
                }

                if ($content) {
                    // Define default url
                    $data['url'] = $data['data']['url'] ?? ContentUrlGenerator::generateUrl($content);
                } else {
                    continue; // old notification on a deleted content
                }

                $transformer = new ContentTransformer();

                if (isset($data['type']['fields'])) {
                    $fields = $data['type']['fields'];
                    $hasDataFrom = false;
                    if (isset($fields['notification-data.from'])) {
                        $hasDataFrom = $fields['notification-data.from'];
                        unset($fields['notification-data.from']);
                    }

                    $transformedData = (array) $transformer->transform($content, $fields);

                    if ($hasDataFrom) {
                        $transformedData = array_merge(['notification-data.from' => $hasDataFrom], $transformedData);
                    }
                }
                if (! isset($data['url'])) {
                    // Check if we asked slug from a reference ?
                    foreach ($transformedData as $neededData) {
                        if (isset($neededData['data']) && count($neededData['data']) > 0) {
                            if (isset($neededData['data'][0]['slug'])) {
                                $data['url'] = $neededData['data'][0]['slug'][0];
                                break;
                            }
                        }
                    }
                }

                if (isset($data['type']['extra_fields']) && $data['type']['extra_fields']) {
                    $transformedExtraData = (array) $transformer->transform($content, $data['type']['extra_fields']);
                }
            }

            // Remove translations uuids
            unset($transformedData['translations']);
            unset($transformedExtraData['translations']);

            $data['model'] = $transformedData;
            $flattened = $this->flattenData($transformedData, $from);
            $flattenedExtra = $this->flattenData($transformedExtraData, $from);

            $data['icon'] = $data['type']['icon'] ?? '';
            $data['text'] = $data['type']['text'] ?? '';
            $data = array_merge($data['data'], $data);
            $data['data'] = (isset($extra) && is_array($extra)) ? array_merge($extra, $flattened) : $flattened;

            $data['data'] = array_filter($data['data']);
            if (! empty($flattenedExtra)) {
                $data['extra_data'] = $flattenedExtra;
            }

            unset(
                $data['notification_type_id'],
                $data['user_uuid'],
                $data['type'],
                $data['notifiable_type'],
                $data['notifiable_uuid'],
                $data['updated_at']
            );

            // Check for var in translation string
            if ($content !== null) {
                $data['text'] = $this->getTranslationKey($data['text'], $content, $data);
            }

            // Note: Arr dot is intend to be a reversible function, so when there is
            // an empty array, it keep it that way BUT we only wants stringable values
            // and [] is not
            $dotedData = array_filter(
                Arr::dot($data),
                function ($value) {
                    return $value !== [];
                }
            );

            $translated = Lang::get($data['text'], $dotedData, $locale);
            if ($translated == $data['text']) {
                $translated = trans($data['text'], $dotedData, $locale);
                if ($translated == $data['text']) {
                    $translated = null;
                }
            }
            if ($translated !== null) {
                // Check if translation is needed
                $matches = null;
                // Special magic translation in translation, don't be lost !
                if (preg_match('#@__([<>\w\.\-\_]+)#', $translated, $matches) === 1) {
                    $toTranslate = str_replace(['<', '>'], '', $matches[1]);
                    if (__($toTranslate, [], $locale) !== $toTranslate) {
                        $translated = str_replace("@__{$matches[1]}", __($toTranslate, [], $locale), $translated);
                    }
                }

                $data['content'] = $translated;
            }

            //':'.$key, ':'.Str::upper($key), ':'.Str::ucfirst($key)
            // If it fails to translate for any reason we don't show notification
            // to avoid weird notification
            // we are matching official php variable regex expression (won't match 8:30 for example)
            if (isset($data['content'])
                && preg_match(
                    '#\s?:([a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]+)(\.[a-zA-Z0-9_\x7f-\xff]+)*#',
                    $data['content']
                ) === 0
            ) {
                $transformed[] = $data;
            } else {
                Log::warning('Notification has a weird content ['.json_encode($data).']');
            }
        }

        return $transformed;
    }

    /**
     * flatten transformed data
     *
     * @param array $transformedData
     * @param Users|null $from
     * @param array $flattened
     * @return array|mixed
     */
    public function flattenData(array $transformedData, ?Users $from, array &$flattened = [])
    {
        foreach ($transformedData as $key => $value) {
            if (Str::startsWith((string) $key, 'notification-data.from') && $from) {
                $transformer = new ContentTransformer();
                $transformedData = (array) $transformer->transform($from, ['firstname', 'lastname']);

                $flattened = $flattened + $this->flattenData($transformedData, $from);
                continue;
            }

            if (is_array($value)) {
                $flattened = $this->flattenData($value, $from, $flattened);
                continue;
            }

            $flattened[] = $value;
        }

        return $flattened;
    }

    /**
     * @param User $user
     * @param array $filters
     * @param bool $notViewed
     * @param bool $notRed
     * @return Collection
     */
    public function getNotifications(User $user, array $filters = [], bool $notViewed = false, bool $notRed = false): Collection
    {
        $rolesIds = RoleService::listUserRoleIds($user);
        $systemNotificationTypes = config('notify.system_notifications', []);

        // Helper pour créer des sous-requêtes dynamiques
        $subQuery = fn ($sql, $bindings = []) => sprintf('(%s)', $sql);

        // Générer les sous-requêtes
        $isRoleNotificationSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notification_role WHERE notification_id = inside_notifications.id AND role_id IN ('.
            collect($rolesIds)->map(fn () => '?')->join(',').') LIMIT 1'
        );

        $isSubscribedSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notifications_subscribers WHERE notification_type_id = inside_notifications.notification_type_id AND user_uuid = ? LIMIT 1'
        );

        $isViewedSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notifications_users_views WHERE notification_id = inside_notifications.id AND view_at IS NOT NULL AND user_uuid = ? LIMIT 1'
        );

        $isRedSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notifications_users_views WHERE notification_id = inside_notifications.id AND read_at IS NOT NULL AND user_uuid = ? LIMIT 1'
        );

        $isUserNotificationSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notifications AS notifs WHERE id = inside_notifications.id AND user_uuid = ? LIMIT 1'
        );

        $isSystemNotificationSql = $subQuery(
            'SELECT COUNT(*) FROM inside_notifications_types WHERE id = inside_notifications.notification_type_id AND type IN ('.
            collect($systemNotificationTypes)->map(fn () => '?')->join(',').') LIMIT 1'
        );

        // Construire la requête principale
        $query = Notification::query()->selectRaw(
            collect([
                'inside_notifications.*',
                "$isRoleNotificationSql AS is_role_notification",
                "$isSubscribedSql AS is_subscribed",
                "$isViewedSql AS is_viewed",
                "$isRedSql AS is_red",
                "$isUserNotificationSql AS is_user_notification",
                "$isSystemNotificationSql AS is_system_notification",
            ])->join(', '),
            array_merge(
                $rolesIds,                         // Pour `is_role_notification`
                [$user->uuid, $user->uuid, $user->uuid, $user->uuid], // Pour `is_subscribed`, `is_viewed`, `is_red`, `is_user_notification`
                $systemNotificationTypes           // Pour `is_system_notification`
            )
        )
            ->whereNotNull('notifiable_type')
            ->where(fn ($query) => $query
                ->whereNull('author_uuid')
                ->orWhere('author_uuid', '!=', $user->uuid)
            )
            ->where(fn ($query) => $query
                ->whereNull('notifiable_langcode')
                ->orWhere('notifiable_langcode', $user->langcode)
            )
            ->when(
                ! isset($filters['type']) || ! is_string($filters['type']) || empty($filters['type']),
                fn ($query) => $query->where('type', '!=', 'activity')
            )
            ->whereRaw(
                collect([
                    "($isUserNotificationSql != 0 OR $isRoleNotificationSql != 0) AND $isSubscribedSql != 0",
                    "$isSystemNotificationSql != 0 AND $isUserNotificationSql != 0",
                ])->join(' OR '),
                array_merge(
                    [$user->uuid],                     // Pour `is_user_notification` dans le HAVING
                    $rolesIds,                         // Pour `is_role_notification`
                    [$user->uuid],                     // Pour `is_subscribed`
                    $systemNotificationTypes,          // Pour `is_system_notification`
                    [$user->uuid]                      // Pour la dernière condition
                )
            )
            ->orderBy('created_at', 'desc');

        // Appliquer les filtres additionnels
        foreach ($filters as $filter => $value) {
            if (! in_array($filter, ['paginate', 'limit', 'offset'], true)) {
                $query->where($filter, $value);
            }
        }

        // Récupérer les notifications avec conditions spéciales
        $notifications = $query->get()
            ->when($notRed, fn (Collection $collection) => $collection->where('is_red', 0))
            ->when($notViewed, fn (Collection $collection) => $collection->where('is_viewed', 0));

        $userCreatedAt = $user->created_at ? Carbon::createFromTimestamp($user->created_at) : Carbon::now();

        return $notifications->filter(function ($notification) use ($userCreatedAt) {
            $notifCreatedAt = Carbon::parse($notification->created_at);

            return $notifCreatedAt->greaterThanOrEqualTo($userCreatedAt);
        });
    }

    /**
     * List notification types
     *
     * @param Request $request
     *
     * @return Collection|LengthAwarePaginator|Paginator|\Illuminate\Pagination\LengthAwarePaginator|Builder[]|null
     */
    public function list(Request $request)
    {
        $filters = ContentHelper::extractFiltersInputFromRequest($request);
        $user = Permission::user();

        if (! $user) {
            return null;
        }

        $collection = $this->getNotifications($user, $filters);

        return ContentHelper::getPaginationFromCollection(
            collection: $collection,
            limit: (int) $request->input('limit', 10),
            page: (int) $request->input('page', 1)
        );
    }

    /**
     * Count user notification not viewed_at
     *
     * @param User $user
     * @param Request $request
     * @return int
     */
    public function count(User $user, Request $request): int
    {
        $filters = [];

        if ($request->has('filters')) {
            $filters = json_decode($request->get('filters'), true) ?? [];
        }

        return $this->getNotifications($user, $filters, true)->count();
    }

    /**
     * Get a notification type
     *
     * @param int $notificationId
     *
     * @return Notification
     */
    public function get(int $notificationId): ?Notification
    {
        return Notification::find($notificationId);
    }

    /**
     * Update a new notification type
     *
     * @param array $data
     * @param int $notificationId
     *
     * @return Notification
     */
    public function update(array $data, int $notificationId): ?Notification
    {
        $notification = $this->get($notificationId);

        if (! is_null($notification)) {
            $notification->update($data);
        }

        return $notification;
    }

    /**
     * @param Notification $notification
     * @param string $userUuid
     * @return bool
     */
    public function isUserNotification(Notification $notification, string $userUuid): bool
    {
        if ($notification->user_uuid === $userUuid) {
            return true;
        }

        $userRoles = User::find($userUuid)->roles->pluck('id');
        $notificationRoles = $notification->roles->pluck('id');

        return $notificationRoles->intersect($userRoles)->isNotEmpty();
    }

    /**
     * @param int $notificationId
     * @param string $userUuid
     * @return Model|null
     */
    public function getOrCreateNotificationUserView(int $notificationId, string $userUuid): ?Model
    {
        $notification = $this->get($notificationId);

        if (! $notification || ! $this->isUserNotification($notification, $userUuid)) {
            return null;
        }

        return NotificationUserView::firstOrCreate(
            [
                'notification_id' => $notificationId,
                'user_uuid' => $userUuid,
            ]
        );
    }

    /**
     * @param User $user
     * @param int $notificationId
     * @return bool
     */
    public function view(User $user, int $notificationId): bool
    {
        $userView = $this->getOrCreateNotificationUserView($notificationId, $user->uuid);

        if (! $userView) {
            return false;
        }

        $userView->view_at = Carbon::now();

        return $userView->save();
    }

    /**
     * @param User $user
     * @param int $notificationId
     * @return bool
     */
    public function read(User $user, int $notificationId): bool
    {
        $userView = $this->getOrCreateNotificationUserView($notificationId, $user->uuid);

        if (! $userView) {
            return false;
        }

        $userView->read_at = Carbon::now();

        return $userView->save();
    }

    /**
     * @param User $user
     * @return int
     */
    public function viewAll(User $user): int
    {
        $notificationIds = $this->getNotifications($user, [], true)->pluck('id'); // only not viewed notifications
        $notificationIds->each(
            function ($notificationId) use ($user) {
                $this->view($user, $notificationId);
            }
        );

        return $notificationIds->count();
    }

    /**
     * @param User $user
     * @return int
     */
    public function readAll(User $user): int
    {
        $notificationIds =
            $this->getNotifications($user, [], false, true)->pluck('id'); // only not red notifications
        $notificationIds->each(
            function ($notificationId) use ($user) {
                $this->read($user, $notificationId);
            }
        );

        return $notificationIds->count();
    }
}
