<?php

declare(strict_types=1);

namespace Inside\Notify\Services;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Queue;
use Inside\Content\Models\Content;
use Inside\Notify\Models\NotificationLog;
use Inside\Notify\Models\NotificationType;
use Inside\Permission\Models\Role;

/**
 * NotificationLogService service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
final class NotificationLogService
{
    private const JOB_NOT_FOUND = 'validation.scheduled.notification.not.found';

    private const JOB_DELETE_SUCCESS = 'validation.scheduled.notification.canceled';

    private const JOB_DELETE_FAIL = 'validation.scheduled.notification.cancel.failed';

    private const STATUS_SCHEDULED = 0;

    private const STATUS_SENT = 1;

    private const STATUS_CANCELED = 2;

    public function getLogs(string $contentUuid, string $contentType): Collection
    {
        return NotificationLog::where([
            'content_uuid' => $contentUuid,
            'content_type' => $contentType,
        ])->orderBy('sent_at')
            ->get()->transform(
                function ($log) {
                    $log->roles = Role::whereIn('id', $log->roles)->get()->pluck('label');
                    $log->sent_at = get_date_in_user_timezone($log->sent_at);
                    $log->created_at = get_date_in_user_timezone($log->created_at);
                    $log->updated_at = get_date_in_user_timezone($log->updated_at);

                    return $log;
                }
            );
    }

    public function logNotification(
        string $channel,
        array $roleIds,
        Content $content,
        NotificationType $notificationType,
        ?string $queuedJobId = null,
        ?Carbon $sentAt = null
    ): NotificationLog {
        $status = (bool) setting('email', 'email_sending_enabled') || $channel !== 'email'
            ? self::STATUS_SENT
            : self::STATUS_CANCELED;

        return NotificationLog::create([
            'queued_job_id' => $queuedJobId,
            'content_type' => $content->content_type,
            'content_uuid' => $content->uuid,
            'channel' => $channel,
            'roles' => $roleIds,
            'sent_at' => $sentAt ?: now(),
            'status' => $sentAt ? self::STATUS_SCHEDULED : $status,
        ]);
    }

    public function markAsSent(string $queuedJobId): void
    {
        /** @var Content|null $notification */
        $notification = NotificationLog::where('queued_job_id', $queuedJobId)->first();

        $status = (bool) setting('email', 'email_sending_enabled') || $notification?->channel !== 'email'
            ? self::STATUS_SENT
            : self::STATUS_CANCELED;

        $notification?->update(['status' => $status, 'sent_at' => now()]);
    }

    public function isProcessable(string $queuedJobId): bool
    {
        return NotificationLog::where([
            'queued_job_id' =>  $queuedJobId,
            'status' => self::STATUS_SCHEDULED,
        ])->exists();
    }

    public function cancelScheduledNotification(int $notificationLogId): array
    {
        $return = [
            'success' => false,
            'message' => Lang::get(self::JOB_NOT_FOUND),
        ];

        $notificationLog = NotificationLog::where([
            'id' => $notificationLogId,
            'status' => self::STATUS_SCHEDULED,
        ])
            ->first();

        if (! $notificationLog) {
            return $return;
        }

        if ($notificationLog->queued_job_id) {
            try {
                $queuedJobId = $notificationLog->queued_job_id;
                $now = now();
                if ($notificationLog->sent_at && $notificationLog->sent_at > $now->addMinutes(5)) {
                    NotificationLog::where(['id' => $notificationLogId])->update(['status' => -1]);
                    $return = [
                        'success' => true,
                        'message' => Lang::get(self::JOB_DELETE_SUCCESS),
                    ];
                }
            } catch (\Exception $exception) {
                Log::error("Failed to remove job from queue: {$exception->getMessage()}");
                $return = [
                    'success' => false,
                    'message' => Lang::get(self::JOB_DELETE_FAIL),
                ];
            }
        }

        return $return;
    }
}
