<?php

namespace Inside\Notify\Providers;

use Illuminate\Notifications\NotificationServiceProvider;
use Illuminate\Support\Facades\Notification;
use Inside\Authentication\Events\AuthenticationLoginEvent;
use Inside\Cache\RateLimiting\Limit;
use Inside\Content\Events\CommentCreatedEvent;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Content\Events\ContentUpdatingEvent;
use Inside\Content\Events\UserMentionedInCommentEvent;
use Inside\Events\InsideBooted;
use Inside\Facades\RateLimiter;
use Inside\Notify\Channels\SmsChannel;
use Inside\Notify\Console\CleanNotificationsCommand;
use Inside\Notify\Console\CleanOutdatedNotificationsCommand;
use Inside\Notify\Console\CleanupCommand;
use Inside\Notify\Console\DeleteOldNotificationsCommand;
use Inside\Notify\Console\DeleteUserNotificationsCommand;
use Inside\Notify\Console\DumpCommand;
use Inside\Notify\Console\InstallOneSignalSDKWorkerAssetsCommand;
use Inside\Notify\Console\SubscribeBaseNotificationsCommand;
use Inside\Notify\Events\CustomNotificationEvent;
use Inside\Notify\Events\NotificationsManualsEvent;
use Inside\Notify\Facades\SendinBlue;
use Inside\Notify\Facades\SMS;
use Inside\Notify\Facades\Twilio;
use Inside\Notify\Http\Middlewares\ContentRoleFormMiddleware;
use Inside\Notify\Http\Middlewares\NotificationRequestRole;
use Inside\Notify\Listeners\AuthenticationLoginListener;
use Inside\Notify\Listeners\CommentCreatedListener;
use Inside\Notify\Listeners\ContentCreatedListener;
use Inside\Notify\Listeners\ContentDeletedListener;
use Inside\Notify\Listeners\ContentUpdatedListener;
use Inside\Notify\Listeners\ContentUpdatingListener;
use Inside\Notify\Listeners\CustomNotificationListener;
use Inside\Notify\Listeners\NotificationManualsListener;
use Inside\Notify\Listeners\PushNotificationListener;
use Inside\Notify\Listeners\ReactionSubscribedListener;
use Inside\Notify\Listeners\ReactionUnsubscribedListener;
use Inside\Notify\Listeners\TestEmailCreatedListener;
use Inside\Notify\Listeners\TestEmailUpdatedListener;
use Inside\Notify\Listeners\UserDisabledListener;
use Inside\Notify\Listeners\UserEnabledListener;
use Inside\Notify\Listeners\UserMentionedListener;
use Inside\Notify\Notifications\InsideNotification;
use Inside\Notify\Services\ExpiredNotificationCleanerService;
use Inside\Notify\Services\ManualNotificationsService;
use Inside\Notify\Services\NotificationService;
use Inside\Notify\Services\NotificationTypeHelper;
use Inside\Notify\Services\OneSignalClientService;
use Inside\Notify\Services\SendinBlueService;
use Inside\Notify\Services\SmsService;
use Inside\Notify\Services\TwilioService;
use Inside\Reaction\Events\ReactionSubscribedEvent;
use Inside\Reaction\Events\ReactionUnsubscribedEvent;
use Inside\Support\EventServiceProvider;
use Inside\User\Events\UserDisabledEvent;
use Inside\User\Events\UserEnabledEvent;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

/**
 * Inside package Notify service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotifyServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        ReactionSubscribedEvent::class => [
            ReactionSubscribedListener::class,
        ],
        ReactionUnsubscribedEvent::class => [
            ReactionUnsubscribedListener::class,
        ],
        ContentCreatedEvent::class => [
            ContentCreatedListener::class,
            TestEmailCreatedListener::class,
        ],
        CommentCreatedEvent::class => [
            CommentCreatedListener::class,
        ],
        ContentDeletedEvent::class => [
            ContentDeletedListener::class,
        ],
        ContentUpdatedEvent::class => [
            ContentUpdatedListener::class,
            TestEmailUpdatedListener::class,
        ],
        ContentUpdatingEvent::class => [
            ContentUpdatingListener::class,
        ],
        AuthenticationLoginEvent::class => [
            AuthenticationLoginListener::class,
        ],
        CustomNotificationEvent::class => [
            CustomNotificationListener::class,
        ],
        UserDisabledEvent::class => [
            UserDisabledListener::class,
        ],
        UserEnabledEvent::class => [
            UserEnabledListener::class,
        ],
        UserMentionedInCommentEvent::class => [
            UserMentionedListener::class,
        ],
        NotificationsManualsEvent::class => [
            NotificationManualsListener::class,
        ],
        InsideBooted::class => [
            PushNotificationListener::class,
        ],
    ];

    protected array $commands = [
        DeleteUserNotificationsCommand::class,
        DeleteOldNotificationsCommand::class,
        SubscribeBaseNotificationsCommand::class,
        DumpCommand::class,
        CleanNotificationsCommand::class,
        CleanupCommand::class,
        InstallOneSignalSDKWorkerAssetsCommand::class,
        CleanOutdatedNotificationsCommand::class,
    ];

    public array $bindings = [
        'notify.type' => NotificationTypeHelper::class,
    ];

    public function boot(): void
    {
        parent::boot();

        $this->loadViewsFrom(__DIR__.'/resources/views', 'notifications');
    }

    /**
     * Register the application services.
     */
    public function register(): void
    {
        try {
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/notify.php', 'notify');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/onesignal.php', 'onesignal');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/sms.php', 'sms');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/scheduler.php', 'scheduler');
        } catch (NotFoundExceptionInterface|ContainerExceptionInterface) {
        }

        $this->app->register(NotificationServiceProvider::class);

        $this->app->middleware([ContentRoleFormMiddleware::class]);
        $this->app->middleware([NotificationRequestRole::class]);

        $this->app->withFacades(
            true,
            [
                Notification::class => 'Notification',
            ]
        );

        RateLimiter::for('notification', function (InsideNotification $notification) {
            return $notification->getNotificationType()->via !== 'sms' ?
                Limit::none() :
                Limit::perMinute(40)->by((string) $notification->getNotificationType()->id);
        });

        $this->app->bind(
            'onesignal.client',
            function ($app) {
                return new OneSignalClientService(
                    $app['config']['onesignal']['app_id'],
                    $app['config']['onesignal']['rest_api_key'],
                    $app['config']['onesignal']['mobile_activated'],
                    $app['config']['onesignal']['desktop_activated'],
                );
            }
        );

        $this->app->bind(
            NotificationService::class,
            function ($app) {
                return new NotificationService();
            }
        );

        $this->app->bind(
            ManualNotificationsService::class,
            function ($app) {
                return new ManualNotificationsService();
            }
        );

        $this->app->singleton(SendinBlueService::class, function ($app) {
            return new SendinBlueService(
                $app['config']['sms']['drivers']['sendinblue'],
                $app['config']['sms']['sender'],
                $app['config']['sms']['mobile_field_name'],
                $app['config']['sms']['limit_per_month']
            );
        });
        $this->app->insideAlias('SendinBlue', SendinBlue::class);

        $this->app->singleton(TwilioService::class, function ($app) {
            return new TwilioService(
                $app['config']['sms']['drivers']['twilio'],
                $app['config']['sms']['sender'],
                $app['config']['sms']['mobile_field_name'],
                $app['config']['sms']['limit_per_month']
            );
        });
        $this->app->insideAlias('Twilio', Twilio::class);

        $this->app->singleton(SmsService::class, function ($app) {
            return new SmsService($app['config']['sms']['default']);
        });
        $this->app->insideAlias('SMS', SMS::class);
        // Add Inside SmsChannel
        Notification::extend(SmsChannel::class, function () {
            return new SmsChannel();
        });

        $this->app->alias('onesignal.client', OneSignalClientService::class);

        if ($this->app->runningInConsole()) {
            $this->app->singleton(
                'command.notif:clean:expired',
                function ($app) {
                    return new CleanOutdatedNotificationsCommand(
                        $app->make(ExpiredNotificationCleanerService::class)
                    );
                }
            );
            $this->commands('command.notif:clean:expired');
        }
    }
}
