<?php

declare(strict_types=1);

namespace Inside\Notify\Notifications;

use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Notify\Channels\MailChannel;
use Inside\Notify\Messages\MailMessage;
use Inside\Notify\Models\NotificationType;
use Inside\Settings\Models\Setting;

/**
 * Mail notification.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class MailNotification extends InsideNotification
{
    /**
     * @throws Exception
     */
    public function __construct(
        protected NotificationType $notificationType,
        protected ?User $user,
        protected ?Content $model,
        protected ?array $data = [],
        protected ?string $theme = null
    ) {
        $isEmailSendingEnabled = Setting::where('key', 'email_sending_enabled')->first()?->value;
        if (is_null($this->user) || ! $isEmailSendingEnabled) {
            if (! $isEmailSendingEnabled) {
                Log::warning('MailNotifications are disabled - mail not sent to '.$this->user?->email);
            }
            throw new Exception('MailNotification needs a user recipient');
        }
        parent::__construct($notificationType, $user, $model, $data);
    }

    public function via(mixed $notifiable): array
    {
        return [MailChannel::class];
    }

    /**
     * @throws Exception
     */
    public function toMail(mixed $notifiable): MailMessage
    {
        return $this->createMailMessage($this->prepareMail());
    }

    /**
     * @throws Exception
     */
    public function createMailMessage(array $mail): MailMessage
    {
        if (is_null($this->user)) {
            throw new Exception('MailNotification needs a user recipient');
        }

        if ($mail['view'] === 'onboardingEmail') {
            $mail['subject'] = $this->getTranslationFromSettings($mail['view'], 'subject', $this->user->langcode, 'courses');
            $mail['data']['onboarding_title'] = $this->getTranslationFromSettings($mail['view'], 'text', $this->user->langcode, 'courses');
            $mail['data']['onboarding_button_text'] = $this->getTranslationFromSettings($mail['view'], 'button_text', $this->user->langcode, 'courses');
        }

        $mailSubject = empty($this->data['test']) ? $mail['subject'] : '[TEST] '.$mail['subject'];
        $mailMessage = (new MailMessage())
            ->markdown(
                'notifications::'.$mail['view'],
                ['locale' => $this->user->langcode, 'data' => $mail['data']]
            )
            ->theme($this->theme)
            ->model($this->model)
            ->recipient($this->user)
            ->subject($mailSubject);

        if ($this->model?->content_type === 'external_notifications' || $mail['view'] !== 'enhancedEmail' && $mail['view'] !== 'reminderEmail') {
            // Intro lines
            foreach ($mail['contents'] as $content) {
                $mailMessage->line($content);
            }
        }
        // Add action
        $mailMessage->action(Lang::get($mail['buttonText'], [], $this->user->langcode), $mail['url']);
        if ($mail['view'] === 'onboardingEmail' || $mail['view'] === 'enhancedEmail' || $this->model?->content_type === 'external_notifications') {
            if (! is_null($this->model) && isset($this->model->image)) {
                $image = url('notify_content_files', [$this->model->content_type, $this->model->uuid]);
                $mailMessage->image($image);
            }
        }

        return $mailMessage;
    }

    /**
     * @return array
     * @throws Exception
     */
    public function prepareMail(): array
    {
        if (is_null($this->user)) {
            throw new Exception('MailNotification needs a user recipient');
        }
        $view = $this->data['extra']['view'] ?? ($this->notificationType->data['view'] ?? 'email');
        $subject = $this->data['extra']['mail']['subject'] ?? ($this->notificationType->data['mail']['subject'] ?? '');
        $text = $this->data['extra']['mail']['text'] ?? ($this->notificationType->data['mail']['text'] ?? '');
        $subtext = $this->data['extra']['mail']['subtext'] ?? ($this->notificationType->data['mail']['subtext'] ?? '');
        $buttonText = $this->data['extra']['mail']['buttonText'] ?? ($this->notificationType->data['mail']['buttonText'] ?? '');
        $from = false;

        if (! is_null($this->user->langcode)) {
            Lang::setLocale($this->user->langcode);
        }

        // Replace magic var
        Log::warning('[subject]'.$subject.' [text]'.$text.' [button]'.$buttonText);
        if (! is_null($this->model)) {
            $subject = $this->getTranslationKey($subject, $this->model, $this->data ?? []);
            $text = $this->getTranslationKey($text, $this->model, $this->data ?? []);
            $subtext = $this->getTranslationKey($subtext, $this->model, $this->data ?? []);
            $buttonText = $this->getTranslationKey($buttonText, $this->model, $this->data ?? []);
        }

        Log::warning('[subject]'.$subject.' [text]'.$text.' [subtext] '.$subtext.' [button]'.$buttonText);

        // Settings $datas
        $data = $this->data ?? [];

        // Recipient info ( usefull to say Hello Mr John DOE )
        // Use :to.lastname for magic laravel string replacement for example
        $recipient = Users::find($this->user->uuid);
        $data['to'] = [
            'uuid' => $recipient->uuid,
            'lastname' => $recipient->lastname,
            'firstname' => $recipient->firstname,
            'fullname' => trim($recipient->firstname.' '.$recipient->lastname),
            'image' => $recipient->image,
        ];

        if (array_key_exists('from', (array) $this->data) && $data['from']) {
            // If data from is set, let's get user infos
            try {
                $from = Users::findOrFail($data['from']);
                $data['author'] = [
                    'uuid' => $from->uuid,
                    'lastname' => $from->lastname,
                    'firstname' => $from->firstname,
                    'fullname' => trim($from->firstname.' '.$from->lastname),
                    'image' => $from->image,
                ];
            } catch (ModelNotFoundException $exception) {
            }
        }

        $data['url'] = $this->getContentUrl();
        Log::info('[MailNotification] Build url from getContentUrl : ('.$data['url'].')', [
            'data' => $data,
        ]);

        $transformer = new ContentTransformer();
        $transformedData = [];

        if (isset($this->notificationType->data['fields'])) {
            $transformedData = $transformer->transform($this->model, $this->notificationType->data['fields']);
        }

        if (! isset($data['url']) && ! empty($transformedData)) {
            // Check if we asked slug from a reference ?
            foreach ($transformedData as $neededData) {
                if (isset($neededData['data']) && count($neededData['data']) > 0) {
                    if (isset($neededData['data'][0]['slug'])) {
                        $data['url'] = $neededData['data'][0]['slug'][0];
                        break;
                    }
                }
            }
            Log::info('[MailNotification] Build url from transformedData ('.$data['url'].')', [
                'data' => $data,
                'transformedData' => $transformedData,
            ]);
        }

        $data = Arr::dot(array_merge($data, $transformedData));
        $data = array_map(
            function ($item) {
                return is_string($item) ? trim($item) : '';
            },
            $data
        );

        $subject = Lang::get($subject, $data, $this->user->langcode);
        $content = Lang::get($text, $data, $this->user->langcode);

        if ($subtext) {
            $content .= "\n".Lang::get($subtext, $data, $this->user->langcode);
        }

        if (! is_null($this->model)) {
            $subject = $this->replaceLastVariables($subject, $this->model);
            $content = $this->replaceLastVariables($content, $this->model);
        }

        if ($from) {
            $subject = str_replace('{author}', $from->firstname.' '.$from->lastname, $subject);
            $content = str_replace('{author}', $from->firstname.' '.$from->lastname, $content);
        }

        $subject = str_replace('{title}', $this->model->title ?? '', $subject);

        $content = str_replace('{title}', $this->model->title ?? '', $content);
        $contents = explode("\n", $content);
        $return = [
            'buttonText' => $buttonText,
            'contents' => $contents,
            'data' => $data,
            'subject' => $subject,
            'url' => url($data['url']),
            'view' => $view,
        ];
        Log::info('[MailNotification] Inspect contents to send:', [
            'contents' => $contents,
            'return' => $return,
        ]);

        return $return;
    }
}
