<?php

namespace Inside\Notify\Notifications;

use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Str;
use Inside\Content\Models\Content;
use Inside\Slug\Facades\ContentUrlGenerator;

trait HasSpecialUrl
{
    /**
     * @param Content $model
     * @param array|string $fields
     * @return string
     */
    protected function getUrlForRelation(Content $model, $fields): string
    {
        if (! is_array($fields)) {
            $fields = [$fields];
        }

        // Try to find one relation, first found is the ONE !
        foreach ($fields as $field) {
            try {
                $field = Str::camel($field);
                if (isset($model->{$field})) {
                    if ($model->{$field} instanceof Collection && $model->{$field}->isNotEmpty()) {
                        $url = ContentUrlGenerator::generateUrl($model->{$field}->first());
                    } else {
                        $url = ContentUrlGenerator::generateUrl($model->{$field});
                    }
                    if (isset($model->uuid)) {
                        $url = $url.'#&d/'.$model->uuid;
                    }

                    return $url ?? '';
                }
            } catch (\Exception $e) {
            }
        }

        return ''; // Not found, redirect to home
    }

    protected function isFolderChildren(string $field): bool
    {
        // check if model is a folder and has a parent who is also a folder
        return ! is_null($this->model)
            && class_to_type($this->model) === $field
            && class_to_type($this->model->{$field}->first()) === $field;
    }

    public function prepareUrl(string $url): string
    {
        if (is_null($this->model)) {
            return '#';
        }
        if (Str::startsWith($url, 'RELATION:')) {
            $fields = explode(',', Str::after($url, 'RELATION:'));

            $url = $this->getUrlForRelation($this->model, $fields);
        } elseif (Str::startsWith($url, 'ROOT_RELATION:')) {
            $fields = explode(',', Str::after($url, 'ROOT_RELATION:'));

            // use root as a start for searching relation
            $root = $this->model;
            while ($root->pid != null) {
                // Going up to the root !
                $root = $root->belongsTo(get_class($this->model), 'pid')->first();
            }

            $url = $this->getUrlForRelation($root, $fields);
        } elseif (Str::startsWith($url, 'URLFIELD:')) {
            $field = Str::after($url, 'URLFIELD:');
            $url = ltrim($this->model->{$field}, '/');
        } elseif (Str::startsWith($url, 'GED:') && ! empty(config('notify.ged.url'))) {
            $rootContentType = config('notify.ged.root_content_type');
            $rootContentUuid = match ($this->model->content_type) {
                $rootContentType => $this->model->uuid,
                default => $this->model->{$rootContentType}?->first()?->uuid,
            };

            if ($rootContentUuid !== null) {
                $url = config('notify.ged.url').'?'.http_build_query([config('notify.ged.query', $rootContentType) => $rootContentUuid]);
            }
        } elseif ($url == 'EDIT') {
            $url = 'edit/'.class_to_type($this->model).'/'.$this->model->uuid;
        } elseif (Str::startsWith($url, 'CUSTOM:')) {
            $callback = explode(':', $url)[1];
            $customUrlCallbacks = config('notify.build_url_callbacks', false);

            if (
                $customUrlCallbacks && isset($customUrlCallbacks[$callback])
                && is_callable(
                    $customUrlCallbacks[$callback]
                )
            ) {
                $url = $customUrlCallbacks[$callback]($this->model);
            }
        }

        return $url;
    }

    public function getContentUrl(): ?string
    {
        if (isset($this->data['url'])) {
            return $this->data['url'];
        }

        if (config('slug.magic_urls')) {
            return ContentUrlGenerator::generateUrl($this->model);
        }

        $url = null;
        // If a specific url is given in data, use this one instead
        if (isset($this->notificationType->data['url'])) {
            $url = $this->prepareUrl($this->notificationType->data['url']);
        } elseif (isset($this->model->slug[0])) {
            $url = $this->model->slug[0];
        }

        return $url;
    }
}
