<?php

namespace Inside\Notify;

use Illuminate\Notifications\AnonymousNotifiable;
use Illuminate\Notifications\Events\NotificationSending;
use Illuminate\Notifications\NotificationSender as BaseNotificationSender;
use Illuminate\Support\Str;
use Inside\Notify\Models\Notification;
use Inside\Notify\Notifications\InsideNotification;
use Inside\Support\Contracts\Translation\HasLocalePreference;
use Inside\Support\Traits\Localizable;

#[\AllowDynamicProperties]
class NotificationSender extends BaseNotificationSender
{
    use Localizable;

    /**
     * The locale to be used when sending notifications.
     *
     * @var string|null
     */
    protected $locale;

    public function __construct($manager, $bus, $events, ?string $locale = null)
    {
        parent::__construct($manager, $bus, $events);
        $this->locale = $locale;
    }

    /**
     * @param $notifiables
     * @param $notification
     * @param  array|null  $channels
     * @return void
     */
    public function sendNow($notifiables, $notification, array $channels = null): void
    {
        $notifiables = $this->formatNotifiables($notifiables);

        $original = clone $notification;

        foreach ($notifiables as $notifiable) {
            if (empty($viaChannels = $channels ?: $notification->via($notifiable))) {
                continue;
            }

            $this->withLocale(
                $this->preferredLocale($notifiable, $notification),
                function () use ($viaChannels, $notifiable, $original) {
                    $notificationId = Str::uuid()->toString();

                    foreach ((array) $viaChannels as $channel) {
                        if (! ($notifiable instanceof AnonymousNotifiable && $channel === 'database')) {
                            $this->sendToNotifiable($notifiable, $notificationId, clone $original, $channel);
                        }
                    }
                }
            );
        }
    }

    /**
     * @param mixed $notifiable
     * @param Notification $notification
     * @return string
     */
    protected function preferredLocale($notifiable, $notification): string
    {
        return $notification->locale ?? $this->locale ?? value(function () use ($notifiable) {
            if ($notifiable instanceof HasLocalePreference) {
                return $notifiable->preferredLocale();
            }
        });
    }

    protected function shouldSendNotification($notifiable, $notification, $channel): bool
    {
        if (method_exists($notification, 'shouldSend') &&
            $notification->shouldSend($notifiable, $channel) === false) {
            return false;
        }

        return $this->events->until(
            new NotificationSending($notifiable, $notification, $channel)
        ) !== false;
    }

    /**
     * @param $notifiables
     * @param $notification
     * @return void
     */
    protected function queueNotification($notifiables, $notification): void
    {
        $notifiables = $this->formatNotifiables($notifiables);

        $original = clone $notification;

        if (! method_exists($original, 'via') || ! $original instanceof InsideNotification) {
            return;
        }

        foreach ($notifiables as $notifiable) {
            $notificationId = Str::uuid()->toString();

            foreach ((array) $original->via($notifiable) as $channel) {
                $notification = clone $original;

                $notification->id = $notificationId;

                if (! is_null($this->locale)) {
                    $notification->locale = $this->locale;
                }

                $queue = $notification->queue;

                if (method_exists($notification, 'viaQueues')) {
                    $queue = $notification->viaQueues()[$channel] ?? null;
                }

                $this->bus->dispatch(
                    (new SendQueuedNotifications($notifiable, $notification, [$channel]))
                        ->onConnection($notification->connection)
                        ->onQueue($queue)
                        ->delay(
                            is_array($notification->delay) ?
                            ($notification->delay[$channel] ?? null)
                            : $notification->delay
                        )
                        ->through(
                            array_merge(
                                method_exists($notification, 'middleware') ? $notification->middleware() : [],
                                $notification->middleware ?? []
                            )
                        )
                );
            }
        }
    }
}
