<?php

namespace Inside\Notify\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Notifications\DatabaseNotification;
use Illuminate\Support\Facades\Event;
use Inside\Content\Models\Contents\Users;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Notify\Events\NotificationCreatedEvent;
use Inside\Notify\Events\NotificationDeletedEvent;
use Inside\Notify\Events\NotificationUpdatedEvent;
use Inside\Permission\Models\Role;

/**
 * Inside Notification model.
 *
 * @category Class
 * @author Maecia <technique@maecia.com>
 * @license http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link http://www.maecia.com/
 * @property int $id
 * @property int $notification_type_id
 * @property string|null $user_uuid
 * @property string $notifiable_type
 * @property string $notifiable_uuid
 * @property string|null $notifiable_langcode
 * @property string|null $type
 * @property string|null $parent
 * @property array $data
 * @property int|null $created_at
 * @property int|null $updated_at
 * @property string|null $author_uuid
 * @property bool $is_role_notification
 * @property bool $is_user_notification
 * @property bool $is_system_notification
 * @property bool $is_subscribed
 * @property-read \Illuminate\Database\Eloquent\Model|\Eloquent $notifiable
 * @property-read \Illuminate\Database\Eloquent\Collection|Role[] $roles
 * @property-read int|null $roles_count
 * @property-read NotificationType $trigger
 * @property-read Users|null $user
 * @property-read \Illuminate\Database\Eloquent\Collection|\Inside\Notify\Models\NotificationUserView[] $view
 * @property-read int|null $view_count
 * @method static \Illuminate\Notifications\DatabaseNotificationCollection|static[] all($columns = ['*'])
 * @method static \Illuminate\Notifications\DatabaseNotificationCollection|static[] get($columns = ['*'])
 * @method static \Inside\Database\Eloquent\Builder|Notification newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|Notification newQuery()
 * @method static \Inside\Database\Eloquent\Builder|Notification query()
 * @method static Builder|Notification whereAuthorUuid($value)
 * @method static Builder|Notification whereCreatedAt($value)
 * @method static Builder|Notification whereData($value)
 * @method static Builder|Notification whereId($value)
 * @method static Builder|Notification whereNotifiableLangcode($value)
 * @method static Builder|Notification whereNotifiableType($value)
 * @method static Builder|Notification whereNotifiableUuid($value)
 * @method static Builder|Notification whereNotificationTypeId($value)
 * @method static Builder|Notification whereParent($value)
 * @method static Builder|Notification whereType($value)
 * @method static Builder|Notification whereUpdatedAt($value)
 * @method static Builder|Notification whereUserUuid($value)
 *
 * @mixin \Eloquent
 * @mixin Builder
 */
#[\AllowDynamicProperties]
class Notification extends DatabaseNotification
{
    use WithEnhancedBuilder;

    /**
     * Is the table id increments
     *
     * @var bool
     */
    public $incrementing = true;

    protected $keyType = 'int';

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'inside_notifications';

    /**
     * The attributes that are guarded.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * Casts the following attributes
     *
     * @var array
     */
    protected $casts = [
        'data' => 'array',
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
    ];

    public static function boot(): void
    {
        parent::boot();

        self::created(
            function ($model) {
                Event::dispatch(new NotificationCreatedEvent($model));
            }
        );

        self::updated(
            function ($model) {
                Event::dispatch(new NotificationUpdatedEvent($model));
            }
        );

        self::deleted(
            function ($model) {
                Event::dispatch(new NotificationDeletedEvent($model));
            }
        );
    }

    /**
     * Type information ( Note: type column has been added afterward ... )
     *
     * @return BelongsTo
     */
    public function trigger(): BelongsTo
    {
        return $this->belongsTo(NotificationType::class, 'notification_type_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(Users::class, 'user_uuid', 'uuid');
    }

    public function view(): HasMany
    {
        return $this->hasMany(
            NotificationUserView::class,
            'notification_id',
            'id'
        );
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(
            Role::class,
            'inside_notification_role',
            'notification_id',
            'role_id'
        );
    }

    /**
     * @return MorphTo
     */
    public function notifiable(): MorphTo
    {
        return $this->morphTo('notifiable', 'notifiable_type', 'notifiable_uuid');
    }

    public function getUserView(string $userUuid): ?NotificationUserView
    {
        return $this->view()->where('user_uuid', $userUuid)->first() ?? null;
    }
}
