<?php

namespace Inside\Notify\Migrations;

use Exception;
use Illuminate\Database\Migrations\Migration;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Models\NotificationTypeRole;
use Inside\Permission\Models\Role;
use InvalidArgumentException;

class ReportMigration extends Migration
{
    /**
     * register a report notification type for $roles
     *
     * @param  array  $roles
     * @param  string  $type
     * @param  bool  $viaWeb
     * @param  bool  $viaEmail
     * @param  string|null  $url
     * @throw InvalidArgumentException
     * @throws Exception
     */
    protected function registerReport(array $roles, string $type, bool $viaWeb, bool $viaEmail, ?string $url = null): void
    {
        if (! $viaEmail && ! $viaWeb) {
            throw new InvalidArgumentException('Vous devez au moins régler la notification via Web ou via Email');
        }

        // Check & delete old report notification
        NotificationType::whereAction('report')->whereModel(type_to_class($type))->whereRole(true)->delete();

        $notificationTypes = [];
        // viaWeb
        if ($viaWeb) {
            $data = $this->getNotificationData($type, 'web', $url);
            $notificationTypes[] =
                NotificationType::firstOrCreate(array_except($data, ['data']), ['data' => $data['data']]);
        }
        // viaEmail
        if ($viaEmail) {
            $data = $this->getNotificationData($type, 'email', $url);
            $notificationTypes[] =
                NotificationType::firstOrCreate(array_except($data, ['data']), ['data' => $data['data']]);
        }

        foreach ($roles as $role) {
            if (is_int($role)) {
                // Find by id
                $role = Role::find($role);
            } elseif (is_string($role)) {
                // Find by name
                $role = Role::where('name', $role)->first();
            } elseif (! $role instanceof Role) {
                continue;
            }
            if (! $role) {
                continue;
            }

            foreach ($notificationTypes as $notificationType) {
                // Register notifications to role
                NotificationTypeRole::firstOrCreate(
                    [
                        'notification_type_id' => $notificationType->id,
                        'role_id'              => $role->id,
                    ]
                );

                // Register users because it is default to true
                $role->users->each(
                    function ($user) use ($notificationType) {
                        NotificationSubscriber::firstOrCreate(
                            [
                                'user_uuid'            => $user->uuid,
                                'notification_type_id' => $notificationType->id,
                            ]
                        );
                    }
                );
            }
        }
    }

    /**
     * Prepare notification type data
     *
     * @param string      $type
     * @param string      $via
     * @param string|null $url
     * @return array
     */
    protected function getNotificationData(string $type, string $via, ?string $url = null): array
    {
        $notification = [
            'via'       => $via,
            'default'   => true,
            'event'     => 'Inside\Notify\Events\CustomNotificationEvent',
            'model'     => type_to_class($type),
            'action'    => 'report',
            'role'      => true,
            'condition' => null,
            'multiple'  => true,
            'language'  => true,
            'profile'   => false,
            'data'      => [
                'title'       => 'notifications.custom.report.<content_type>.title',
                'description' => 'notifications.custom.report.<content_type>.description',
                'icon'        => 'report',
                'text'        => 'notifications.custom.report.<content_type>.text',
                'fields'      => ['authors', 'title', 'content_type'],
                'url'         => $url,
            ],
        ];

        $mail = [
            'mail' => [
                'subject'    => 'notifications.custom.report.<content_type>.mail.subject',
                'text'       => 'notifications.custom.report.<content_type>.mail.content',
                'buttonText' => 'notifications.custom.report.<content_type>.mail.buttonText',
            ],
        ];

        return $via === 'email' ? array_merge_recursive($notification, ['data' => $mail]) : $notification;
    }
}
