<?php

namespace Inside\Notify\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\User;
use Inside\Content\Events\UserMentionedInCommentEvent;
use Inside\Content\Models\Contents\Comments;
use Inside\Notify\Models\NotificationType;
use Inside\Slug\Facades\ContentUrlGenerator;

/**
 * Comment created listener
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class UserMentionedListener extends BaseNotificationListener implements ShouldQueue
{
    public ?string $queue;

    public function __construct()
    {
        $this->queue = get_low_priority_queue_name();
    }

    public function handle(UserMentionedInCommentEvent $event): void
    {
        if ($event->commentType != 'comments') {
            return;
        }

        Log::debug('UserMentionedListener');

        /** @var ?User $subscriber */
        $subscriber = User::find($event->userUuid);

        if (is_null($subscriber)) {
            return;
        }

        // Comment created can trigger two notification type familly
        // a comment attach to a content
        // a comment attach to an other content
        $types = NotificationType::where('model', Comments::class)->where(
            function ($query) {
                $query->whereIn('action', ['userMentioned', 'userMentionedInResponse']);
            }
        )->get();
        $comment = call_user_func(type_to_class($event->commentType).'::find', $event->commentUuid);
        foreach ($types as $type) {
            // Check comment obvious conditions

            // Let's check condition on parent content type
            $conditionTypes = [];

            if (! empty($type->condition)) {
                $conditions = explode('|', $type->condition);
                $newConditions = [];
                foreach ($conditions as $condition) {
                    [$name, $value] = explode(':', $condition, 2);
                    if ($name == 'parent_type') {
                        $conditionTypes = explode(',', $value);
                    } else {
                        $newConditions[] = $condition;
                    }
                }
                $type->condition = implode('|', $newConditions);
            }

            // Let's get the content that comment is attached to
            $parents = $comment->getParentsIfExist(
                count($conditionTypes) > 0 ? ['content_type' => $conditionTypes] : []
            );

            if (empty($parents) && $comment->parent == null) {
                continue; // It's too soon, pivot not set yet OR not correct content type
            }

            // Merge parents
            if ($comment->parent) {
                $parents['comments'] = [$comment->parent];
            }

            if ($type->action === 'userMentionedInResponse' && ! array_key_exists('comments', $parents)) {
                continue;
            }

            $scrollComment = $comment;
            $model = $parent = null;

            switch ($type->action) {
                case 'userMentionedInResponse': // use parent comment, front can't scroll to comment responses
                    if (array_key_exists('comments', $parents) && ! empty($parents['comments']) && count($parents) > 1) {
                        $parent = Arr::first($parents['comments']) ?? null;
                        $scrollComment = Arr::first($parents['comments']) ?? null;

                        foreach ($parents as $content_type => $contentParents) {
                            if ($content_type != 'comments' && ! empty($contentParents)) {
                                $model = Arr::first($contentParents);
                                break;
                            }
                        }
                    }
                    break;
                case 'userMentioned':
                    if (array_key_exists('comments', $parents) && ! empty($parents['comments']) && count($parents) > 1) {
                        continue 2;
                    }

                    foreach ($parents as $content_type => $contentParents) {
                        if ($content_type != 'comments' && ! empty($contentParents)) {
                            $model = $parent = Arr::first($contentParents);
                            break;
                        }
                    }
                    break;
                default:
                    continue 2;
            }

            $this->when = $this->getInterval($type, $comment);
            $notificationData = [
                'from' => $comment->author,
                'comment' => $comment,
                'parentComment' => $scrollComment,
                'url' => (ContentUrlGenerator::generateUrl($model) ?? '').'#&d/'.$scrollComment->uuid,
            ];

            if ($event->userUuid === $comment->author) {
                return;
            }

            $this->route = $subscriber;

            // Check additional conditions
            // Now conditions are checked against first parent
            if ($parent && $model && $this->checkConditions($type, $parent, $subscriber)
                && $this->checkUniqueness(
                    $type,
                    $parent,
                    $subscriber
                )
            ) {
                $this->notify(
                    $type,
                    $subscriber,
                    $model,
                    $notificationData
                );
            }
        }
    }
}
