<?php

namespace Inside\Notify\Listeners;

use Illuminate\Contracts\Queue\ShouldQueue;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Models\Contents\Users;
use Inside\Facades\Package;
use Inside\Notify\Models\NotificationSubscriber;
use Inside\Notify\Models\NotificationType;
use Inside\Workflow\Facades\Workflow;

/**
 * Content created listener
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ContentCreatedListener extends BaseNotificationListener implements ShouldQueue
{
    /**
     * @param  ContentCreatedEvent  $event
     */
    public function handle(ContentCreatedEvent $event): void
    {
        $model = $event->model;

        // A user has been created, let's subscribe him to default notification types
        if (get_class($model) === Users::class) {
            NotificationType::where('default', true)->each(
                function ($notificationType) use ($model) {
                    NotificationSubscriber::firstOrCreate(
                        [
                            'user_uuid' => $model->uuid,
                            'notification_type_id' => $notificationType->id,
                        ]
                    );
                }
            );
        }

        if (Package::has('inside-workflow')) {
            /** @var User $user */
            $user = User::find($model->author);
            if (Workflow::isContentInWorkflow($model->content_type) && $user && ! $user->hasAnyRole('super_administrator')) {
                return;
            }
        }

        $types = NotificationType::where('model', get_class($model))->where(
            function ($query) {
                $query->where('action', 'create')->orWhere('action', 'createOrUpdate');
            }
        )->get();

        foreach ($types as $type) {
            // notifications groupées TODO v1 need improve
            if ($type->role && $type->via !== 'email') {
                $this->when = $this->getInterval($type, $model);
                // on en crée une seule et pas une par role (pour éviter les doublons)
                $this->route = $type->roles->first();
                if (! $this->checkConditions($type, $model)) {
                    continue;
                }
                $this->notify($type, null, $model, ['from' => $model->author], $type->roles->pluck('id')->toArray());
            } else {
                $type->subscribers->each(
                    function ($subscriber) use ($type, $model) {
                        if ($subscriber->uuid === $model->author && $type->type != 'activity') {
                            return;
                        }

                        $this->when = $this->getInterval($type, $model);
                        $this->route = $subscriber;

                        if (! $this->checkConditions($type, $model, $subscriber) ||
                            ! $this->checkUniqueness($type, $model, $subscriber)
                        ) {
                            return;
                        }
                        $roles = $type->role ? $type->roles->pluck('name')->toArray() : [];
                        $this->notify($type, $subscriber, $model, ['from' => $model->author], $roles);
                    }
                );
            }
        }
    }
}
