<?php

namespace Inside\Notify\Jobs;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Jobs\Job;
use Inside\Notify\Models\Notification;
use Inside\Permission\Models\Role;

/**
 * Class CleanupRoleNotifications
 */
class CleanupRoleNotifications extends Job
{
    public function __construct(
        protected Role $role
    ) {
    }

    /**
     * handle the job
     */
    public function handle(): void
    {
        $toDelete = [];
        if ($this->role->notifications === null) {
            return; // No notification
        }

        // Clean old notification, lost etc... for $this->user
        $this->role->notifications->each(
            function (Notification $notification) use (&$toDelete) {
                try {
                    // Notifiable does not exist anymore
                    if ($notification->notifiable === null
                        || // Wrong created_at
                        get_date((int) $notification->created_at) === null
                        || // Notification is activity is older than 3 month
                        ($notification->type == 'activity'
                            && get_date((int) $notification->created_at)->isBefore(
                                Carbon::now()->subMonths(config('notify.lifetimes.activity', 3))
                            ))
                        || // Notification is global is older than 6 month
                        ($notification->type == 'global'
                            && get_date((int) $notification->created_at)->isBefore(
                                Carbon::now()->subMonths(config('notify.lifetimes.global', 6))
                            ))
                    ) {
                        $toDelete[] = $notification->id;
                    }
                } catch (\Throwable | \Exception $e) {
                    $toDelete[] = $notification->id;
                    Log::debug('[Notify::Cleanup](roles) '.$e->getMessage());
                }
                if (count($toDelete) > 1000) {
                    DB::table('inside_notifications')->whereIn('id', $toDelete)->delete();
                    $toDelete = [];
                }
            }
        );
        DB::table('inside_notifications')->whereIn('id', $toDelete)->delete();
    }
}
