<?php

namespace Inside\Notify\Http\Middlewares;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Content\Facades\ContentHelper;
use Inside\Http\Middleware\TransformResponseAndRequest;
use Inside\Notify\Events\NotificationsManualsEvent;

/**
 * Class NotificationRequestRole
 */
class NotificationRequestRole extends TransformResponseAndRequest
{
    /**
     * @var string[][]
     */
    protected array $paths = [
        'POST' => [
            '#^api/v1/content/([^/]+)$#',
        ],
        'PUT' => [
            '#^api/v1/content/([^/]+)/([\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12})$#',
        ],
    ];

    protected array|string $channels;

    protected ?array $roleIds = [];

    protected ?string $contentType;

    protected function patchRequest(Request $request): Request
    {
        $this->roleIds = ContentHelper::getAndRemoveQuery($request, 'notifications_roles');
        $this->channels = ContentHelper::getAndRemoveQuery($request, 'notifications_vias') ?? [];

        return $request;
    }

    /**
     * @param Request $request
     * @param mixed $response
     * @param array $data
     * @return mixed
     */
    protected function patch(Request $request, mixed $response, array $data = []): mixed
    {
        if (config('notify.manual_notifications.enabled') !== true || ! $response instanceof JsonResponse) {
            return $response;
        }
        $this->contentType = $data[1];
        if (count($data) === 2) {
            return $this->patchOnStore($request, $response);
        }

        return $this->patchOnUpdate($request, $response);
    }

    /**
     * dispatch event on content creation
     *
     * @param Request $request
     * @param JsonResponse $response
     * @return JsonResponse
     */
    protected function patchOnStore(Request $request, JsonResponse $response): JsonResponse
    {
        /** @var array $responseData
         */
        $responseData = json_decode_response($response);
        $roleIds = $this->roleIds;
        $channels = $this->channels;
        if (! isset($responseData['uuid'])) {
            // Failed to create content
            return $response;
        }
        if (! is_array($channels) || empty($channels) || ! is_array($roleIds)) {
            return $response;
        }
        NotificationsManualsEvent::dispatch($responseData['uuid'], $roleIds, $channels, $this->contentType);

        return $response;
    }

    /**
     * dispatch event on content update
     *
     * @param Request $request
     * @param JsonResponse $response
     * @return JsonResponse
     */
    protected function patchOnUpdate(Request $request, JsonResponse $response): JsonResponse
    {
        $responseData = json_decode_response($response);
        $roleIds = $this->roleIds;
        $channels = $this->channels;

        if (! isset($responseData['uuid']) || $roleIds === null) {
            return $response;
        }

        if (! is_array($channels) || empty($channels) || ! is_array($roleIds)) {
            return $response;
        }

        NotificationsManualsEvent::dispatch($responseData['uuid'], $roleIds, $channels, $this->contentType);

        return $response;
    }
}
