<?php

namespace Inside\Notify\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Exceptions\NotificationTypeValidatorException;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Services\NotificationSubscriberService;
use Inside\Notify\Services\NotificationTypeService;
use InvalidArgumentException;

/**
 * Notification Type controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotificationTypeController
{
    /**
     * The notification type service.
     *
     * @var NotificationTypeService
     */
    protected $service;

    /**
     * Create a new notification type controller instance.
     *
     * @param  NotificationTypeService  $service
     * @return void
     */
    public function __construct(NotificationTypeService $service)
    {
        $this->service = $service;
    }

    /**
     * Get notification type
     *
     * @param  int  $notificationTypeId
     * @return JsonResponse
     */
    public function get(int $notificationTypeId): JsonResponse
    {
        return new JsonResponse([
            'data' => $this->service->get($notificationTypeId),
        ]);
    }

    /**
     * List notification types
     *
     * @param  Request  $request
     * @return JsonResponse
     */
    public function list(Request $request): JsonResponse
    {
        return new JsonResponse([
            'data' => $this->service->list($request),
        ]);
    }

    /**
     * Add a notification type
     *
     * @param  Request  $request
     * @return NotificationType|null
     * @throws NotificationTypeValidatorException
     */
    public function create(Request $request): ?NotificationType
    {
        return $this->service->create($request->all());
    }

    /**
     * Update a notification type
     *
     * @param  Request  $request
     * @param  int  $notificationTypeId
     * @return NotificationType|null
     * @throws NotificationTypeValidatorException
     */
    public function update(Request $request, int $notificationTypeId): ?NotificationType
    {
        return $this->service->update($request->all(), $notificationTypeId);
    }

    /**
     * Delete a notification type
     *
     * @param  int  $notificationTypeId
     * @throws Exception
     */
    public function delete(int $notificationTypeId): void
    {
        $this->service->delete($notificationTypeId);
    }

    /**
     * Subscribe user to a notification type if exists, else, create it
     *
     * @param  Request  $request
     * @return array
     * @throws InvalidArgumentException|Exception
     */
    public function unsubscribe(Request $request): array
    {
        $datas = $request->all();

        if (! is_array($datas) || count($datas) === 0) {
            throw new InvalidArgumentException('missing subscription data');
        }

        reset($datas);
        // array_key_first 7.3
        if (! is_int(key($datas))) {
            $datas = [$datas];
        }

        $notificationTypes = [];
        foreach ($datas as $data) {
            foreach ($data as &$value) {
                if (in_array($value, ['true', 'false'])) {
                    $value = ($value === 'true');
                }
            }

            $notificationType = NotificationType::where(array_except($data, ['data']))->first();

            if ($notificationType) {
                $subscriberService = new NotificationSubscriberService();
                $subscriberService->unsubscribe($notificationType->id);
            }
            $notificationTypes[] = $notificationType;
        }

        return $notificationTypes;
    }

    /**
     * Subscribe user to a notification type if exists, else, create it
     *
     * @param  Request  $request
     * @return array
     * @throws InvalidArgumentException|NotificationTypeValidatorException
     */
    public function subscribe(Request $request): array
    {
        $datas = $request->all();

        if (! is_array($datas) || count($datas) === 0) {
            throw new InvalidArgumentException('missing subscription data');
        }
        reset($datas);
        if (! is_int(key($datas))) {
            $datas = [$datas];
        }

        $notificationTypes = [];
        foreach ($datas as $data) {
            /** @var NotificationType $notificationType */
            $notificationType = $this->service->create($data);

            $subscriberService = new NotificationSubscriberService();
            $subscriberService->subscribe($notificationType->id);
            $notificationTypes[] = $notificationType;
        }

        return $notificationTypes;
    }

    /**
     * @param  Request  $request
     * @param  string  $uuid
     * @return JsonResponse
     * @throws NotificationTypeValidatorException
     */
    public function followUser(Request $request, string $uuid): JsonResponse
    {
        if (! $uuid || ! Users::find($uuid)) {
            throw new InvalidArgumentException('You should provide a valid user uuid');
        }

        $type = $request->get('type') ?? 'news';

        $types = [
            [
                'via' => 'web',
                'default' => false,
                'event' => 'Inside\Content\Events\ContentCreatedEvent',
                'model' => type_to_class($type),
                'action' => 'create',
                'condition' => 'author:'.$uuid.'|status:1',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'data' => [
                    'title' => 'notifications.create.content.'.$type.'.title',
                    'description' => 'notifications.create.content.'.$type.'.description',
                    'icon' => 'news',
                    'text' => 'notifications.create.content.'.$type.'.text',
                    'fields' => [
                        ['authors' => ['firstname', 'lastname']],
                        'title',
                    ],
                ],
            ],
            [
                'via' => 'email',
                'default' => false,
                'event' => 'Inside\Content\Events\ContentCreatedEvent',
                'model' => type_to_class($type),
                'action' => 'create',
                'condition' => 'author:'.$uuid.'|status:1',
                'multiple' => false,
                'language' => true,
                'profile' => false,
                'data' => [
                    'title' => 'notifications.create.content.'.$type.'.title',
                    'description' => 'notifications.create.content.'.$type.'.description',
                    'mail' => [
                        'subject' => 'notifications.create.content.'.$type.'.subject',
                        'text' => 'notifications.create.content.'.$type.'.content',
                        'buttonText' => 'notifications.create.content.'.$type.'.buttonText',
                    ],
                    'fields' => [
                        ['authors' => ['firstname', 'lastname']],
                        'title',
                    ],
                ],
            ],
        ];

        foreach ($types as $data) {
            if ($data['via'] === 'email' && ! config('notify.email_following_users')) {
                continue;
            }
            /** @var NotificationType $notificationType */
            $notificationType = $this->service->create($data);
            $subscriberService = new NotificationSubscriberService();
            $subscriberService->subscribe($notificationType->id);
        }

        return new JsonResponse(['success' => true]);
    }

    /**
     * @param  Request  $request
     * @param  string  $uuid
     * @return JsonResponse
     * @throws Exception
     */
    public function unfollowUser(Request $request, string $uuid): JsonResponse
    {
        if (! $uuid || ! Users::find($uuid)) {
            throw new InvalidArgumentException('You should provide a valid user uuid');
        }

        $type = $request->get('type') ?? 'news';

        $notificationTypes = NotificationType::where('model', type_to_class($type))
            ->where('condition', 'author:'.$uuid.'|status:1')->get();

        if ($notificationTypes->isEmpty()) {
            throw new Exception('no notification to unsubscribe from');
        }

        foreach ($notificationTypes as $notificationType) {
            $subscriberService = new NotificationSubscriberService();
            $subscriberService->unsubscribe($notificationType->id);
        }

        return new JsonResponse(['success' => true]);
    }
}
