<?php

namespace Inside\Notify\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Services\NotificationSubscriberService;
use InvalidArgumentException;

/**
 * Subscriber controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotificationSubscriberController
{
    /**
     * The notification type service.
     *
     * @var NotificationSubscriberService
     */
    protected $service;

    /**
     * Create a new notification subscriber controller instance.
     *
     * @param  NotificationSubscriberService  $service
     * @return void
     */
    public function __construct(NotificationSubscriberService $service)
    {
        $this->service = $service;
    }

    /**
     * Subscribe to notifications
     */
    public function subscribe(int $notificationTypeId): JsonResponse
    {
        return new JsonResponse(['status' => $this->service->subscribe($notificationTypeId)]);
    }

    /**
     * Unsubscribe to notifications
     * @throws Exception
     */
    public function unsubscribe(int $notificationTypeId): JsonResponse
    {
        return new JsonResponse(['status' => $this->service->unsubscribe($notificationTypeId)]);
    }

    /**
     * @param Request $request
     * @param int $notificationTypeId
     * @return JsonResponse
     */
    public function listSubscribers(Request $request, int $notificationTypeId): JsonResponse
    {
        $notificationType = NotificationType::findOrFail($notificationTypeId);

        return new JsonResponse($this->service->listSubscribers($request, $notificationType));
    }

    /**
     * Add subscribers
     *
     * @param Request $request
     * @param int $notificationTypeId
     * @return JsonResponse
     */
    public function addSubscribers(Request $request, int $notificationTypeId): JsonResponse
    {
        $notificationType = NotificationType::findOrFail($notificationTypeId);

        $uuids = $request->get('uuids', []);
        if (! is_array($uuids) || empty($uuids)) {
            throw new InvalidArgumentException('Missing uuids');
        }
        $this->service->addSubscribers($notificationType, $uuids);

        return new JsonResponse(
            [
                'success' => true,
            ]
        );
    }

    /**
     * Remove subscribers
     *
     * @param Request $request
     * @param int $id
     * @return JsonResponse
     */
    public function removeSubscribers(Request $request, int $id): JsonResponse
    {
        $notificationType = NotificationType::findOrFail($id);

        $uuids = $request->get('uuids', []);
        if (! is_array($uuids) || empty($uuids)) {
            throw new InvalidArgumentException('Missing uuids');
        }
        $this->service->removeSubscribers($notificationType, $uuids);

        return new JsonResponse(
            [
                'success' => true,
            ]
        );
    }
}
