<?php

namespace Inside\Notify\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\Auth;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Notify\Events\CustomNotificationEvent;
use Inside\Notify\Facades\NotificationCache;
use Inside\Notify\Models\Notification;
use Inside\Notify\Models\NotificationType;
use Inside\Notify\Notifications\WebNotification;
use Inside\Notify\Services\NotificationService;
use Inside\Support\Str;
use Psy\Util\Json;

/**
 * Notifications controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class NotificationController
{
    public function __construct(
        protected NotificationService $notificationService
    ) {
    }

    /**
     * Index page for the notifications.
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     */
    public function list(Request $request): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();
        $notifications = $this->notificationService->list($request);
        $transformed = $this->notificationService->transform($notifications);
        $data = [];

        if ($notifications instanceof LengthAwarePaginator) {
            $data = $notifications->toArray();
        }

        $counter = $this->notificationService->count($user, $request);

        return new JsonResponse(array_merge($data, ['counter' => $counter, 'data' => $transformed]));
    }

    /**
     * Mark one notification as read.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function read(int $id): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        return new JsonResponse(['success' => $this->notificationService->read($user, $id)]);
    }

    /**
     * Mark all notifications as read.
     *
     * @return JsonResponse
     */
    public function readAll(): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        return new JsonResponse(['updated' => $this->notificationService->readAll($user)]);
    }

    /**
     * Mark one notification as viewed.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function view(int $id): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        return new JsonResponse(['success' => $this->notificationService->view($user, $id)]);
    }

    /**
     * Mark all notifications as view.
     *
     * @return JsonResponse
     */
    public function viewAll(): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();

        return new JsonResponse(['updated' => $this->notificationService->viewAll($user)]);
    }

    /**
     * Send a notification to the user
     *
     * @param  Request  $request
     * @return JsonResponse
     */
    public function dispatch(Request $request): JsonResponse
    {
        return new JsonResponse([
            'success' => CustomNotificationEvent::dispatch($request->all()),
        ]);
    }

    /**
     * Get notifications for a content type of uuid
     *
     * @param string $action
     * @param string $type
     * @param string $uuid
     * @param string|null $days
     * @return array
     */
    public function getForAContent(string $action, string $type, string $uuid, string $days = null): array
    {
        $model = 'Inside\Content\Models\Contents\\'.Str::studly($type);
        // Check content existence or visibility
        call_user_func($model.'::findOrFail', $uuid);
        $user = Auth::user();

        $notifications = Notification::where('notifiable_type', $model)->where('notifiable_uuid', $uuid)->whereHas(
            'type',
            function ($query) use ($action, $model) {
                return $query->where('action', (string) trim($action))->where('model', $model);
            }
        )->when(
            $days,
            function ($query, $days) {
                return $query->where('created_at', '>', Carbon::now()->subDays($days)->toDateTimeString());
            }
        )->get();

        return $notifications->filter(
            function ($notification) use ($user) {
                return array_key_exists('from', $notification->data) && ($notification->data['from'] == $user->uuid);
            }
        )->toArray();
    }

    public function test(): void
    {
        /** @var User $me */
        $me = Auth::user();

        $notificationType = NotificationType::firstOrCreate(
            [
                'via' => 'web',
                'default' => false,
                'event' => null,
                'model' => Users::class,
                'type' => 'global',
                'condition' => '',
                'action' => 'test',
                'multiple' => true,
                'language' => false,
                'profile' => false,
            ],
            [
                'data' => [
                    'title' => 'Notification de test',
                    'icon' => 'test',
                    'text' => 'Test pour :firstname :lastname <:email>',
                    'fields' => [
                        'firstname',
                        'lastname',
                        'email',
                    ],
                ],
            ]
        );

        if ($me->information instanceof Content) {
            $me->notify(new WebNotification($notificationType, $me, $me->information, []));
        }
    }
}
