<?php

namespace Inside\Notify\Console;

use DateInterval;
use DateTime;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Artisan;
use Inside\Notify\Models\Notification;

/**
 * Command to delete old notifications.
 *
 * Example:
 *     > php artisan notif:delete-outdated --interval=P2M
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class DeleteOldNotificationsCommand extends Command
{
    public const EXIT_CODE__FAILED = -1;

    public const EXIT_CODE__SUCCESS = 1;

    /**
     * @var string
     */
    protected $name = 'notif:delete-outdated {--interval=P1M}';

    /**
     * @var string
     */
    protected $description = 'Delete outdated notifications';

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'notif:delete-outdated
                            {--interval=P1M : (default = P1M) DateInterval format (P1M, ...}. Will delete all notifications older than Now - interval.';

    /**
     * @inheritDoc
     */
    public function handle(): int
    {
        /** @var string $interval */
        $interval = $this->option('interval');
        if (empty($interval)) {
            $this->error('--interval option cannot be empty');

            return self::EXIT_CODE__FAILED;
        }

        $nowDT = new DateTime('NOW');
        $expiredDT = $nowDT->sub(new DateInterval($interval));

        $this->info('Removing outdated notifications (older than '.$expiredDT->format('Y-m-d').')');

        try {
            $countDeletedNotifications = Notification::where('created_at', '<', $expiredDT->format('Y-m-d'))
                ->delete();
        } catch (\Exception $e) {
            $this->error($e->getMessage());

            return self::EXIT_CODE__FAILED;
        }

        $this->info($countDeletedNotifications.' notifications deleted.');

        return self::EXIT_CODE__SUCCESS;
    }
}
