<?php

namespace Inside\Menu\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Inside\Content\Validation\Rules\ContentExistsRule;
use Inside\Content\Validation\Rules\LinkRule;
use Inside\Menu\Contracts\Menu;
use Inside\Menu\Models\MenuLink;
use Inside\Menu\Validation\Rules\MenuParentExists;
use Laravel\Lumen\Routing\Controller as BaseController;
use Symfony\Component\HttpFoundation\File\Exception\FileNotFoundException;
use Symfony\Component\HttpFoundation\File\File;

/**
 * Menu controller.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class Controller extends BaseController
{
    protected function checkArguments(string $menuName, string $uuid = null): void
    {
        $table = menu_type_to_table($menuName);

        if (! Schema::hasTable($table)) {
            abort(404, 'This menu does not exist');
        }

        if ($uuid && ! call_user_func(menu_type_to_class($menuName).'::find', $uuid)) {
            abort(404, 'This menu link does not exist');
        }
    }

    /**
     * Validation rules for menu link creation and update
     *
     * @param array $data
     * @return array
     */
    protected function getValidationRules(array $data): array
    {
        if (array_key_exists('parent_uuid', $data) || array_key_exists('weight', $data)) {
            return [
                'parent_uuid' => ['required', 'array', new MenuParentExists()],
                'weight' => 'required|integer',
            ];
        }

        $rules = [
            'title' => 'required|max:255',
            'status' => 'required|boolean',
            'link_type' => 'required|string|in:internal_page,external_link,unclickable_link',
            'link' => ['required_unless:link_type,unclickable_link'],
            'display' => 'required|string|in:title_only,icon_title,icon_only,image,button',
            'icon' => 'required_if:display,icon_title,icon_only',
            'image' => 'required_if:display,image|image',
            'color' => 'nullable|string|size:7',
            'page' => 'nullable|string',
            'page_params' => 'nullable|string',
            'open_in_new_tab' => 'nullable|boolean',
            'hide_on_mobile' => 'boolean',
        ];

        if (isset($data['link'])) {
            if (is_array($data['link'])) {
                $rules['link'][] = new ContentExistsRule();
            } else {
                $rules['link'][] = new LinkRule();
            }
        }

        return $rules;
    }

    /**
     * Cast image as a file
     *
     * @param Request $request
     * @param array $data
     * @return void
     */
    protected function handleImage(Request &$request, array &$data)
    {
        if (! array_key_exists('image', $data) || ! is_string($data['image'])) {
            return;
        }

        try {
            $data['image'] = new File(Storage::disk('local')->path($data['image']));
            $request['image'] = $data['image'];
        } catch (FileNotFoundException) {
        }
    }

    /**
     * Format and send response
     */
    protected function success(Menu $menu, MenuLink $menuLink = null): JsonResponse
    {
        $json = [
            'success' => true,
        ];

        if ($menuLink) {
            $menuLink = $menuLink->toArray();
            if (has_menu_link('link', $menuLink)) {
                $menu->transformLink($menuLink);
            }

            $json['data'] = $menuLink;
        }

        return new JsonResponse($json);
    }
}
