<?php

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Log;

if (! function_exists('get_date')) {
    /**
     * @param int|string|DateTimeInterface|Carbon $date
     * @param string|null $format
     * @return Carbon|null
     */
    function get_date(int|string|DateTimeInterface|Carbon $date, ?string $format = null): ?Carbon
    {
        $clone = is_object($date) ? clone $date : $date;

        try {
            $tz = config('app.timezone');

            if ($clone instanceof Carbon) {
                return $clone->setTimezone($tz);
            }

            if ($clone instanceof DateTimeInterface) {
                return Carbon::createFromTimestamp($clone->getTimestamp(), $tz);
            }

            if (is_int($clone)) {
                return Carbon::createFromTimestamp($clone, $tz);
            }

            if ((string) (int) $clone === $clone && ($clone <= PHP_INT_MAX) && ($clone >= ~PHP_INT_MAX)) {
                return Carbon::createFromTimestamp($clone, $tz);
            }

            if (is_null($format) && data_get(date_parse($clone), 'error_count', 0) === 0) {
                return Carbon::createFromTimeString($clone, $tz);
            }

            if (! is_null($format)) {
                $new = Carbon::createFromFormat($format, $clone, $tz);
                if (! $new) {
                    throw new Exception("Can't create date from the given format !");
                }

                return str($format)->contains(':') ? $new : $new->startOfDay();
            }
        } catch (Exception $e) {
            Log::error("[Inside i18n] {$e->getMessage()}");
        }

        return null;
    }
}

if (! function_exists('get_date_in_user_timezone')) {
    /**
     * Get date in user timezone
     *
     * @param int|string|DateTimeInterface|Carbon $date
     * @param string|null $format
     * @return Carbon|null
     */
    function get_date_in_user_timezone(int|string|DateTimeInterface|Carbon $date, ?string $format = null): ?Carbon
    {
        return get_date($date, $format)?->setTimezone(config('app.default_user_timezone'));
    }
}
