<?php

declare(strict_types=1);

namespace Inside\I18n\Translator;

use Illuminate\Contracts\Translation\Loader;
use Illuminate\Support\Facades\App;
use Illuminate\Translation\Translator as LaravelTranslator;
use Inside\I18n\Repositories\TranslationRepository;

/**
 * Class Translator
 *
 * Inside translator
 */
final class Translator extends LaravelTranslator
{
    /**
     * Translator repository
     *
     * @var TranslationRepository|mixed
     */
    protected $repository;

    /**
     * Translator constructor.
     *
     * @param Loader $loader
     * @param string $locale
     */
    public function __construct(Loader $loader, $locale)
    {
        parent::__construct($loader, $locale);

        $this->repository = app()->make(TranslationRepository::class);
    }

    /**
     * Get the translation for the given key.
     *
     * This function is an override of the function parent::get()
     * that loads specific groups before the group '*'
     *
     *  /!\ Be careful with Laravel future updates /!\
     */
    private function getTranslation(string $key, array $replace = [], ?string $locale = null, bool $fallback = true): string | array
    {
        $locale = $locale ?: $this->locale;

        [$namespace, $group, $item] = $this->parseKey($key);

        $locales = $fallback ? $this->localeArray($locale) : [$locale];

        foreach ($locales as $locale) {
            $line = $this->getLine($namespace, $group, $locale, $item, $replace);

            if (! is_null($line)) {
                return $line;
            }
        }

        $this->load('*', '*', $locale);

        $line = $this->loaded['*']['*'][$locale][$key] ?? null;

        return $this->makeReplacements($line ?: $key, $replace);
    }

    /**
     * get a translation ( used by trans for example )
     *
     * @param string $key
     * @param array $replace
     * @param string|null $locale
     * @param bool $fallback
     * @return array|string|null
     */
    public function get($key, array $replace = [], $locale = null, $fallback = true)
    {
        $locale = $locale ?: $this->locale;
        if (empty($locale)) {
            $locale = App::getLocale();
        }
        if ($locale === '__') { // Special locale __ is used by translation system to force "no translation"
            return $key;
        }

        // Get $line from our Translation sources
        $line = $this->getTranslation($key, $replace, $locale, false);

        // In development environment, if translation is not found add $key as a missing translation to our system
        if (App::environment() && $line === $key) {
            $this->addMissingTranslation($key);

            // Reload with our newly added missing translation
            $line = $this->getTranslation($key, $replace, $locale, $fallback);
        }

        return $line;
    }

    /**
     * get translation from json
     *
     * @param string $key
     * @param array $replace
     * @param string $locale
     * @return array|string|null
     */
    public function getFromJson($key, array $replace = [], $locale = null)
    {
        // Inside use getFromJson everywhere instead of using trans() helper
        // TODO: use trans every where getFromJson should only retrieve translation from ... json
        // use trans result here so that inside is still translating back
        return $this->get($key, $replace, $locale);
    }

    /**
     * Add a missing key
     *
     * @param $key
     */
    protected function addMissingTranslation(string $key): void
    {
        [$namespace, $group, $key] = $this->parseKey($key);
        if ($namespace && $group && $key) {
            $this->repository->addMissingTranslation($namespace, $group, $key);
        }
    }

    /**
     * Force set loaded translation
     * $loaded is an array like [$namespace][$group][$locale] = [$key => $text]
     *
     * @param array $loaded
     */
    public function setLoaded(array $loaded): void
    {
        $this->loaded = $loaded;
    }
}
