<?php

declare(strict_types=1);

namespace Inside\I18n\Services;

use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Facades\Package;
use Inside\I18n\Models\CachedTranslation;
use Inside\I18n\Models\Translation as TranslationModel;
use Inside\I18n\Repositories\LanguageRepository;
use Inside\I18n\Repositories\TranslationRepository;
use Throwable;

/**
 * Class TranslationsService
 */
final class TranslationsService
{
    public function __construct(
        protected TranslationRepository $translationRepository,
        protected LanguageRepository $languageRepository
    ) {
    }

    public function getTranslationsByFiles(string $locale): Collection
    {
        $translations = collect();

        // Get lumen translations
        $translations = $translations->merge(
            $this->translationRepository->getArrayTranslations(
                'vendor/maecia-fork/lumen-framework/resources/lang',
                'lumen',
                $locale
            )->mapWithKeys(
                function ($translation, $key) use ($locale) {
                    return [$key => 'vendor/maecia/inside/i18n/resources/lang/'.$locale.'.json'];
                }
            )
        );

        // Get core back translations
        $translations = $translations->merge(
            $this->translationRepository->getArrayTranslations('vendor/maecia/inside/core/resources/lang', 'inside', $locale)
                ->mapWithKeys(
                    function ($translation, $key) use ($locale) {
                        return [
                            $key => 'vendor/maecia/inside/core/resources/custom/lang/'.$locale
                                .'.json',
                        ];
                    }
                )
        );

        // Get customs
        $translations = $translations->merge(
            $this->translationRepository->getJsonTranslations(
                'vendor/maecia/inside/core/resources/custom/lang/',
                $locale
            )->mapWithKeys(
                function ($translation, $key) use ($locale) {
                    return [
                        $key => 'vendor/maecia/inside/core/resources/custom/lang/'.$locale.'.json',
                    ];
                }
            )
        );

        // Get modules translations
        foreach (Package::list() as $packageName => $information) {
            if ($information->getType() !== 'inside' || ! $information->hasLanguage()) {
                continue;
            }
            $translations = $translations->merge(
                $this->translationRepository->getJsonTranslations(
                    'vendor/'.$information->getName().'/resources/lang',
                    $locale
                )->mapWithKeys(
                    function ($translation, $key) use ($packageName, $locale) {
                        return [
                            $key => 'vendor/'.$packageName.'/resources/lang/'.$locale.'.json',
                        ];
                    }
                )
            );
        }

        // Front
        $translations = $translations->merge(
            $this->translationRepository->getFrontTranslations(
                [
                    'locale' => $locale,
                ]
            )->mapWithKeys(
                function ($translation, $key) use ($locale) {
                    return [$key => 'vendor/maecia/inside/lang/'.$locale.'.json'];
                }
            )
        );

        return $translations;
    }

    public function getOverridesByFile(): Collection
    {
        $overrides = collect();
        foreach ($this->languageRepository->allLanguages() as $locale => $name) {
            $overrides = $overrides->merge(
                $this->getTranslationsByFiles($locale)->intersectByKeys(
                    $this->translationRepository->getOverrideItems(
                        [
                            'state' => 'translated',
                            'draft' => false,
                            'exportable' => true,
                            'locale' => $locale,
                        ]
                    )->pluck('text', 'code')
                )->mapToGroups(
                    function ($translationFile, $key) use ($locale) {
                        return [$translationFile => [$key => trans($key, [], $locale)]];
                    }
                )->mapWithKeys(
                    function ($item, $key) {
                        /**
                         * @var Collection $item
                         * @var string $key
                         */
                        return [$key => $item->collapse()];
                    }
                )
            );
        }

        return $overrides;
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function loadCachedTranslations(): void
    {
        if (CachedTranslation::count() == 0) {
            // We need cached translations
            $this->rebuildCachedTranslations();
        }
    }

    /**
     * Get all translations from any source using correct overloading order
     */
    public function getAllTranslationsFromCore(): Collection
    {
        if (! Cache::tags(['translations'])->has('inside.i18n.translations.core')) {
            $translations = collect();
            foreach ($this->languageRepository->allLanguageIds() as $locale => $id) {
                $localeTranslations = collect();

                // TODO: reverse order front should be able to override automatics etc...
                // Get front translations
                $localeTranslations = $localeTranslations->merge(
                    $this->translationRepository->getFrontTranslations(
                        ['locale' => $locale]
                    )
                );

                // Get core translations
                $localeTranslations = $localeTranslations->merge(
                    $this->translationRepository->getCoreTranslations(['locale' => $locale])
                );
                $translations[$locale] = $localeTranslations;
            }
            Cache::tags(['translations'])->forever(
                'inside.i18n.translations.core',
                $translations
            );
        }

        return Cache::tags(['translations'])->get('inside.i18n.translations.core');
    }

    /**
     * Reconstruction du cache
     *
     * @throws Exception|Throwable
     */
    public function rebuildCachedTranslations(): void
    {
        $lock = Cache::lock('rebuild-cached-translations', 10);
        if (! $lock->get()) {
            Log::debug('already rebuilding cached translation');

            return;
        }
        Cache::tags(['translations'])->flush();
        CachedTranslation::truncate();

        // Remove empty translations from database who override core translations
        TranslationModel::query()->whereNull('text')->delete();

        // Prepare cache
        $cache = collect();
        $languages = $this->languageRepository->allLanguageIds();
        $coreTranslations = $this->getAllTranslationsFromCore();

        foreach ($this->translationRepository->getAllKnownKeys() as $key => $translation) {
            foreach ($languages as $locale => $languageId) {
                $text = $coreTranslations[$locale][$key]->text ?? null;
                $type = $translation['type'] ?? 'self';
                if ($type === 'database' && $text === null) {
                    $key = strtolower($key);
                    $type = 'self';
                }
                $cacheKey = trim($languageId.'-'.$translation['namespace'].'-'.$translation['group'].'-'.$key);
                $cache[$cacheKey] = [
                    'language_id' => $languageId,
                    'type' => $type,
                    'namespace' => $translation['namespace'],
                    'group' => $translation['group'],
                    'key' => $key,
                    'text' => null,
                    'original' => $text,
                    'translated' => false,
                ];
            }
        }

        foreach ($this->languageRepository->allLanguageIds() as $locale => $languageId) {
            $translations = $this->translationRepository->getAllTranslationsFromAnySource($locale);
            foreach ($translations as $key => $translation) {
                $cacheKey = $languageId.'-'.$translation->namespace.'-'.$translation->group.'-'.$key;

                if ($cache->has($cacheKey)) {
                    $type = $translation->type;
                    if ($type === 'database' && $translation->text === null) {
                        $type = 'self';
                    }

                    $cachedKey = $cache[$cacheKey];
                    $cachedKey['type'] = $type;
                    $cachedKey['text'] = $translation->text;
                    $cachedKey['translated'] = ($translation->text !== null && $translation->key != $translation->text);
                    $cache[$cacheKey] = $cachedKey;
                } else {
                    Log::debug(
                        '[[WARNING]] Trying to save unknown key ['.$languageId.'-'.$translation->namespace.'-'
                        .$translation->group.'-'.$key.']'
                    );
                }
            }
        }
        DB::transaction(function () use ($cache) {
            foreach ($cache->chunk(500) as $cachedTranslation) {
                try {
                    DB::table('inside_cached_translations')->insert($cachedTranslation->toArray());
                } catch (Exception $exception) {
                    Log::error('Failed to insert cached translation ['.$exception->getMessage().']');
                    throw $exception;
                }
            }
        }, 10);
        $lock->release();
    }

    /**
     * Import translation from data to overrides
     *
     * @param  array  $data
     * @return array
     */
    public function importTranslationsFromData(array $data): array
    {
        $result = [
            'created' => 0,
            'updated' => 0,
            'imported' => 0,
            'failed' => [],
        ];

        foreach ($data as $translationData) {
            if (empty($translationData->key)) {
                continue;
            }
            // Find translation
            $translation = $this->translationRepository->get($translationData->key, $translationData->locale);
            try {
                if ($translation !== null) {
                    if ($translation->namespace != $translationData->namespace
                        || $translation->group != $translationData->group
                    ) {
                        $result['failed'][] = $translationData;
                        continue;
                    }
                    if ($this->translationRepository->updateTranslation($translation->id, $translationData->text)) {
                        $result['updated']++;
                    } else {
                        $result['failed'][] = $translationData;
                    }
                } else {
                    // Prepare data
                    if ($translationData->text !== null
                        && $this->translationRepository->addTranslation(
                            $translationData->locale,
                            $translationData->namespace,
                            $translationData->group,
                            $translationData->short,
                            $translationData->text
                        )
                    ) {
                        $result['created']++;
                    } else {
                        $result['failed'][] = $translationData;
                    }
                }
            } catch (Exception $e) {
                $result['failed'][] = $translationData;
            }
        }

        return $result;
    }

    /**
     * getOverrideItems
     *
     * @param  array  $filters
     * @return mixed
     */
    public function getOverrideItems(array $filters)
    {
        return $this->translationRepository->getOverrideItems($filters);
    }
}
