<?php

declare(strict_types=1);

namespace Inside\I18n\Repositories;

use Exception;
use Illuminate\Database\DatabaseManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Application;
use Inside\Content\Facades\ContentHelper;
use Inside\Facades\Package;
use Inside\I18n\Exceptions\LanguageDoesNotExistsException;
use Inside\I18n\Exceptions\TranslationAlreadyLockedException;
use Inside\I18n\Exceptions\TranslationAlreadyUnlockedException;
use Inside\I18n\Exceptions\TranslationIsLockedException;
use Inside\I18n\Exceptions\TranslationKeyIsInvalidException;
use Inside\I18n\Facades\Translation;
use Inside\I18n\Models\CachedTranslation;
use Inside\I18n\Models\Language;
use Inside\I18n\Models\Translation as TranslationModel;
use Inside\Support\Str;
use Symfony\Component\Finder\Finder;

/**
 * Class TranslationRepository
 */
final class TranslationRepository extends AbstractRepository
{
    /**
     * @var DatabaseManager|mixed
     */
    protected $database;

    /**
     * @var Language
     */
    protected $defaultLocale;

    /**
     * @var Application
     */
    protected $app;

    /**
     * @var LanguageRepository
     */
    protected $languageRepository;

    /**
     * TranslationRepository constructor.
     *
     * @param TranslationModel $model
     * @param Application $app
     * @param LanguageRepository $languageRepository
     */
    public function __construct(TranslationModel $model, Application $app, LanguageRepository $languageRepository)
    {
        $this->model = $model;
        $this->app = $app;
        $this->languageRepository = $languageRepository;
        $this->defaultLocale = $languageRepository->getLanguage($app['config']->get('app.locale'));
        $this->database = $app['db'];
    }

    /**
     * Get an override translation by $key in $locale
     *
     * @param  string  $key
     * @param  string  $locale
     * @return TranslationModel|null
     */
    public function get(string $key, string $locale): ?TranslationModel
    {
        return $this->getModel()->keyed($key)->in($locale)->first();
    }

    /**
     * Load translation from database to be used has a translation loader source
     *
     * @param string $locale
     * @param string|null $namespace
     * @param string $group
     * @return array
     */
    public function loadSource(string $locale, ?string $namespace, string $group): array
    {
        return $this->getModel()->in($locale)->whereNamespace($namespace)->whereDraft(false)->whereNotNull('text')
            ->whereGroup($group)->get()->keyBy(
                'key'
            )->pluck('text', 'key')->toArray();
    }

    /**
     * get all hardcoded translations
     *
     * @return Collection
     */
    public function getHardcodedItems(): Collection
    {
        $translation = collect();

        // Preload lang ( from php array files )
        foreach (['validation', 'notifications', 'auth', 'pagination', 'password'] as $group) {
            $translation->merge(Lang::get($group));
        }

        // From json files
        foreach (Lang::get('*') as $key => $value) {
            if (! strpos($key, '.')) {
                continue;
            }
            [$group, $key] = explode('.', $key, 2);
            if (! $translation->has($group)) {
                $translation[$group] = collect();
            }
            $translation[$group][$key] = $value;
        }

        return $translation;
    }

    /**
     * Mark translation id as exportable
     *
     * @param  string  $key
     * @param  string  $locale
     * @return bool
     */
    public function markAsExporable(string $key, string $locale): bool
    {
        $model = $this->getModel();
        $translation = $model::keyed($key)->in($locale)->firstOrFail();
        if ($translation->isExportable()) {
            return false;
        }
        $translation->markAsExportable();

        return true;
    }

    /**
     * Unmark translation as exportable
     *
     * @param  string  $key
     * @param  string  $locale
     * @return bool
     */
    public function unmarkAsExporable(string $key, string $locale): bool
    {
        $model = $this->getModel();
        $translation = $model::keyed($key)->in($locale)->firstOrFail();
        if (! $translation->isExportable()) {
            return false;
        }
        $translation->unmarkAsExportable();

        return true;
    }

    /**
     * List overrides translation from database
     *
     * @param  array  $filters
     * @return mixed
     */
    public function getOverrideItems(array $filters)
    {
        $model = $this->getModel()->newModelQuery();
        $allowedFilters = ['namespace', 'group', 'draft', 'exportable'];

        foreach ($allowedFilters as $filter) {
            if (array_key_exists($filter, $filters) && ! empty($filters[$filter])) {
                $model->where($filter, $filters[$filter]);
            }
        }

        if (isset($filters['locale'])) {
            $model->in($filters['locale']);
        }

        if (array_key_exists('query', $filters) && ! empty($filters['query'])) {
            $model->where(
                function ($query) use ($filters) {
                    $query->where('key', 'like', '%'.$filters['query'].'%')->orWhere(
                        'text',
                        'like',
                        '%'.$filters['query'].'%'
                    );
                }
            );
        }

        if (isset($filters['state'])) {
            if ($filters['state'] === 'untranslated') {
                $model->whereNull('text');
            } elseif ($filters['state'] === 'translated') {
                $model->whereNotNull('text');
            }
        }

        $limit = null;
        $paginate = array_key_exists('paginate', $filters) && (bool) $filters['paginate'];
        if (! $paginate) {
            if (array_key_exists('limit', $filters) && (int) $filters['limit'] > 0) {
                $model->limit((int) $filters['limit']);
            }
            if (array_key_exists('offset', $filters) && (int) $filters['limit'] > 0) {
                $model->limit((int) $filters['limit']);
            }
        }
        if ($paginate) {
            return tap(
                $model->paginate($filters['limit'] ?? 10),
                function ($paginatedOverrides) {
                    return $paginatedOverrides->getCollection()->transform(
                        function ($override) {
                            return $this->transformOverrideForFront($override);
                        }
                    );
                }
            );
        } else {
            $data = $model->get();

            return $data->transform(
                function ($override) {
                    return $this->transformOverrideForFront($override);
                }
            );
        }
    }

    protected function transformOverrideForFront(TranslationModel $override): TranslationModel
    {
        $override['locale'] = $override->locale;
        $override['short'] = $override->key;
        $override['key'] = $override->code;
        unset($override['created_at'], $override['updated_at']);

        return $override;
    }

    /**
     * Add a translation
     *
     * @param  string  $locale
     * @param  string  $namespace
     * @param  string  $group
     * @param  string  $key
     * @param  string  $value
     * @return bool
     * @throws Exception
     */
    public function addTranslation(string $locale, string $namespace, string $group, string $key, string $value): bool
    {
        $languageId = $this->languageRepository->getLanguage($locale)->id;
        TranslationModel::create(
            [
                'language_id' => $languageId,
                'namespace' => $namespace,
                'group' => $group,
                'key' => $key,
                'text' => $value,
                'draft' => false, // TODO: draft is not supported yet
            ]
        );
        CachedTranslation::create(
            [
                'language_id' => $languageId,
                'namespace' => $namespace,
                'group' => $group,
                'key' => $key,
                'text' => $value,
                'exportable' => false,
                'translated' => $value !== null && $value != $key,
            ]
        );

        return true;
    }

    /**
     * Update a translation
     *
     * @param  int  $translationId
     * @param  string  $text
     * @return bool
     * @throws Exception
     */
    public function updateTranslation(int $translationId, string $text)
    {
        $translation = TranslationModel::findOrFail($translationId);
        if ($translation->isLocked()) {
            throw TranslationIsLockedException::keyed($translation->code);
        }
        $translation->text = $text;

        return $translation->save() && $this->resetCache();
    }

    /**
     * Database override store keys as short ones
     *
     * @param  string  $fullKey
     * @return string
     */
    protected function getDatabaseKey(string $fullKey): string
    {
        $namespace = '*';
        $group = null;
        $key = $fullKey;
        if (strpos($key, '::') !== false) {
            [$namespace, $key] = explode('::', $key, 2);
        }
        if (strpos($key, '.') !== false) {
            [$group, $key] = explode('.', $key, 2);
        }

        return $key;
    }

    /**
     * @param  string  $fullKey
     * @return string
     */
    protected function getFrontKey(string $fullKey): string
    {
        $namespace = '*';
        $key = $fullKey;
        if (strpos($key, '::') !== false) {
            [$namespace, $key] = explode('::', $key, 2);
        }

        return $key;
    }

    /**
     * @param array|string $keys
     * @param string $locale
     * @return Collection
     * @throws Exception
     */
    public function deleteOverridedTranslationsByKey($keys, string $locale): Collection
    {
        if (! is_array($keys)) {
            $keys = [$keys];
        }
        if (empty($keys)) {
            throw new Exception('No valid key provided');
        }
        foreach ($keys as $key) {
            $translation = TranslationModel::keyed($key)->in($locale)->first();
            if (! $translation) {
                throw new Exception('key ['.$key.'] is invalid or not an override');
            }
            $translation->delete();
        }
        Translation::rebuildCachedTranslations();
        $translations = collect();
        foreach ($keys as $key) {
            $translations[$key] = CachedTranslation::keyed($key)->in($locale)->first();
        }

        return $translations;
    }

    /**
     * Update/create all translations fora given $key
     *
     * @param string $key
     * @param array $translations
     * @return Builder|\Illuminate\Database\Eloquent\Collection|CachedTranslation
     * @throws TranslationKeyIsInvalidException
     */
    public function updateTranslationForKey(string $key, array $translations)
    {
        Translation::loadCachedTranslations();
        $masterTranslation = CachedTranslation::keyed($key)->first();
        if ($masterTranslation === null) {
            throw TranslationKeyIsInvalidException::named($key);
        }
        foreach ($translations as $locale => $text) {
            $text = iClean(trim($text));
            $translation = CachedTranslation::keyed($key)->in($locale)->first();
            $language = $this->languageRepository->getLanguage($locale);
            if (! $language->id) {
                continue;
            }
            if (! $translation || (! in_array($translation->type, ['database', 'self']) && $translation->text != $text)) {
                TranslationModel::create(
                    [
                        'key' => $this->getDatabaseKey($key),
                        'language_id' => $language->id,
                        'namespace' => $masterTranslation->namespace,
                        'group' => $masterTranslation->group,
                        'text' => $text,
                        'draft' => false,
                        'exportable' => false,
                    ]
                );
                if (! $translation) {
                    CachedTranslation::create(
                        [
                            'key' => $key,
                            'language_id' => $language->id,
                            'type' => 'database',
                            'namespace' => $masterTranslation->namespace,
                            'group' => $masterTranslation->group,
                            'text' => $text,
                            'translated' => $text !== null,
                            'exportable' => false,
                        ]
                    );
                } else {
                    $translation->text = $text;
                    $translation->translated = $text !== null;
                    $translation->type = 'database';
                    $translation->save();
                }
                continue;
            }
            if ($translation->text == $text) {
                continue; // No change
            }
            $translation->text = $text;
            $translation->type = 'database';
            $translation->translated = $text !== null;
            $translation->save(); // Update cache

            $translation = TranslationModel::keyed($key)->in($locale)->firstOrFail();
            $translation->text = $text;
            $translation->draft = false;
            $translation->save();
        }

        return CachedTranslation::keyed($key)->get();
    }

    /**
     * Delete a translation
     *
     * @param int $translationId
     * @return bool
     * @throws TranslationIsLockedException
     */
    public function deleteTranslation(int $translationId)
    {
        $translation = TranslationModel::findOrFail($translationId);
        if ($translation->isLocked()) {
            throw TranslationIsLockedException::keyed($translation->code);
        }

        return $translation->delete() && $this->resetCache();
    }

    /**
     * Reset cache because it is outdated
     *
     * @return bool
     */
    public function resetCache(): bool
    {
        CachedTranslation::truncate();
        Cache::tags('translations')->flush();

        return true;
    }

    /**
     * lock a translation
     *
     * @param int $translationId
     * @return mixed
     * @throws TranslationAlreadyLockedException
     */
    public function lockTranslation(int $translationId)
    {
        $translation = TranslationModel::findOrFail($translationId);
        if ($translation->isLocked()) {
            throw TranslationAlreadyLockedException::keyed($translation->code);
        }

        return $translation->lock();
    }

    /**
     * unlock a translation
     *
     * @param int $translationId
     * @return mixed
     * @throws TranslationAlreadyUnlockedException
     */
    public function unlockTranslation(int $translationId)
    {
        $translation = TranslationModel::findOrFail($translationId);
        if (! $translation->isLocked()) {
            throw TranslationAlreadyUnlockedException::keyed($translation->code);
        }

        return $translation->unlock();
    }

    /**
     * validate a translation
     *
     * @param int $translationId
     * @return bool
     */
    public function validateTranslation(int $translationId): bool
    {
        $translation = TranslationModel::findOrFail($translationId);
        if (! $translation->isValid()) {
            return false;
        }

        return $translation->validate();
    }

    /**
     * invalidate a translation
     *
     * @param  int  $translationId
     * @return bool
     */
    public function invalidateTranslation(int $translationId): bool
    {
        $translation = TranslationModel::find($translationId);
        if ($translation === null) {
            return false;
        }
        if (! $translation->isValid()) {
            return false;
        }

        return $translation->invalidate();
    }

    /**
     * Create a missing translation to the database system
     *
     * @param  string  $namespace
     * @param  string  $group
     * @param  string  $key
     */
    public function addMissingTranslation(string $namespace, string $group, string $key): void
    {
        TranslationModel::firstOrCreate(
            [
                'language_id' => $this->defaultLocale->id,
                'namespace' => $namespace,
                'group' => $group,
                'key' => $key,
            ],
            [
                'draft' => false,
            ]
        );
    }

    /**
     * Get existing namespaces for $locale
     *
     * @param string $locale
     * @return Collection
     */
    public function getNamespacesForLanguage(string $locale): Collection
    {
        return TranslationModel::in($locale)->select('namespace')->distinct()->get();
    }

    /**
     * Get existing groups for $locale
     *
     * @param string $locale
     * @return Collection
     */
    public function getGroupsForLanguage(string $locale): Collection
    {
        return TranslationModel::in($locale)->select('group')->distinct()->get()->map(
            function ($translation) {
                return $translation->group;
            }
        );
    }

    public function getNamespaceTranslationsFor(string $locale): Collection
    {
        $translations = $this->languageRepository->getLanguage($locale)->translations()->get()->groupBy('namespace');

        return $translations->map(
            function ($translations) {
                return $translations->mapWithKeys(
                    function ($translation) {
                        return [$translation->group => [$translation->key => $translation->value]];
                    }
                );
            }
        );
    }

    public function getGroupTranslationsFor(string $locale): Collection
    {
        $translations = $this->languageRepository->getLanguage($locale)->translations()->get()->groupBy('group');

        return $translations->map(
            function ($translations) {
                return $translations->mapWithKeys(
                    function ($translation) {
                        return [$translation->key => $translation->value];
                    }
                );
            }
        );
    }

    /**
     * Get translation for $key
     *
     * @param string $key
     * @return Collection
     * @throws TranslationKeyIsInvalidException
     */
    public function getTranslations(string $key): Collection
    {
        Translation::loadCachedTranslations();

        $translations = collect();

        foreach (CachedTranslation::keyed($key)->get() as $translation) {
            $date =
                $translation->core ? ($translation->core->updated_at ?? $translation->core->created_at) : null;
            $translations[] = [
                'key' => $key,
                'locale' => $translation->language->locale,
                'type' => $translation->type,
                'namespace' => $translation->namespace,
                'group' => $translation->group,
                'text' => $translation->text,
                'translated' => $translation->translated,
                'date' => $date !== null ? $date->timestamp : null,
            ];
        }
        if ($translations->isEmpty()) {
            throw TranslationKeyIsInvalidException::named($key);
        }

        return $translations;
    }

    /**
     * Get all known groups
     *
     * @return Collection
     */
    protected function getAllKnownGroups(): Collection
    {
        if (! Cache::tags(['translations'])->has('inside.i18n.groups')) {
            $translations = $this->getAllTranslationsFromAnySource('fr');

            Cache::tags(['translations'])->forever(
                'inside.i18n.groups',
                $translations->mapToDictionary(
                    function ($item) {
                        return [$item->group => $item];
                    }
                )->keys()
            );
        }

        return Cache::tags(['translations'])->get('inside.i18n.groups');
    }

    /**
     * Get all known translations ( slow query )
     *
     * @return Collection
     */
    protected function getAllKnownTranslations(): Collection
    {
        if (! Cache::tags(['translations'])->has('inside.i18n.translations')) {
            $collect = collect();
            foreach ($this->languageRepository->allLanguages() as $locale => $text) {
                $translations = $this->getAllTranslationsFromAnySource($locale);
                $collect = $collect->merge($translations);
            }
            Cache::tags(['translations'])->forever('inside.i18n.translations', $collect);
        }

        return Cache::tags(['translations'])->get('inside.i18n.translations');
    }

    /**
     * Get all known translation keys from sources
     *
     * @return Collection
     */
    public function getAllKnownKeys(): Collection
    {
        if (! Cache::tags(['translations'])->has('inside.i18n.keys')) {
            $collect = collect();
            foreach ($this->languageRepository->allKnownLanguages() as $locale => $text) {
                $translations = $this->getAllTranslationsFromAnySource($locale);
                $collect = $collect->merge($translations);
            }
            Cache::tags(['translations'])->forever(
                'inside.i18n.keys',
                $collect->map(
                    function ($translation, $key) {
                        return [
                            'group' => $translation->group,
                            'namespace' => $translation->namespace,
                            'type' => $translation->type,
                        ];
                    }
                )
            );
        }

        return Cache::tags(['translations'])->get('inside.i18n.keys');
    }

    /**
     * Get all known keys by groups
     *
     * @param array $filters
     * @return Collection
     */
    protected function getAllKnownKeysByGroup(array $filters): Collection
    {
        $cacheKey = 'inside.i18n.keys_by_group'.'.'.$filters['locale'];
        if (isset($filters['query'])) {
            $cacheKey .= '.'.md5($filters['query']);
        }
        if (! Cache::tags(['translations'])->has($cacheKey)) {
            $collect = collect();
            foreach ($this->languageRepository->allLanguages() as $locale => $text) {
                $translations = $this->getAllTranslationsFromAnySource($locale);
                if (isset($filters['query'])) {
                    $translations = $this->filterTranslations(
                        [
                            'query' => $filters['query'],
                        ],
                        $translations
                    );
                }
                foreach ($translations as $key => $translation) {
                    if (! $collect->has($translation->group)) {
                        $collect[$translation->group] = collect();
                    }
                    if (! $collect[$translation->group]->has($key)) {
                        $collect[$translation->group][$key] = collect([$locale]);
                    } else {
                        $collect[$translation->group][$key][] = $locale;
                    }
                }
            }
            Cache::tags(['translations'])->forever($cacheKey, $collect);
        }

        return Cache::tags(['translations'])->get($cacheKey);
    }

    /**
     * Get all groups
     *
     * @param array $filters
     * @return LengthAwarePaginator|Collection
     * @throws Exception
     */
    public function getAllGroups(array $filters)
    {
        $locale = $this->getLocaleFromFilters($filters);
        if ($locale === null) {
            $locale = App::getLocale();
        }
        // Add front languages
        Lang::addJsonPath(cms_base_path('vendor/maecia/inside/i18n/front/lang'));
        Lang::addJsonPath(cms_base_path('vendor/maecia/inside/lang'));

        $groups = $this->getAllKnownKeysByGroup($filters)->map(
            function ($group, $key) use ($locale) {
                $result = [
                    'translated' => 0,
                    'untranslated' => 0,
                    'key' => $key,
                    'label' => __('translations.groups.'.$key, [], $locale),
                ];
                foreach ($group as $key => $translated) {
                    if ($translated->contains($locale)) {
                        $result['translated']++;
                    } else {
                        $result['untranslated']++;
                    }
                }

                return $result;
            }
        );
        // Collection::sortKeys does not exists YET !
        $items = $groups->all();
        ksort($items, SORT_REGULAR);
        $groups = collect($items);

        $limit = null;
        $paginate = array_key_exists('paginate', $filters) && (bool) $filters['paginate'];
        if (array_key_exists('limit', $filters) && (int) $filters['limit'] > 0) {
            $limit = (int) $filters['limit'];
        }

        if (! $paginate) {
            if (array_key_exists('offset', $filters) && (int) $filters['offset'] > 0) {
                $groups = $groups->skip((int) $filters['offset']);
            }
            if ($limit !== null) {
                $groups = $groups->take($limit);
            }

            return $groups;
        } else {
            if ($limit === null) {
                $limit = 10; // Default limit
            }
            // Prepare pagination
            $currentPage = LengthAwarePaginator::resolveCurrentPage();
            $currentPageItems = $groups->slice(($currentPage - 1) * $limit, $limit);

            return new LengthAwarePaginator($currentPageItems, $groups->count(), $limit, $currentPage);
        }
    }

    /**
     * get available filters for translation index
     *
     * @return array
     */
    public function getFilters(): array
    {
        return [
            'namespace' => CachedTranslation::pluck('namespace')->unique()->mapWithKeys(
                function ($namespace) {
                    return [$namespace => __('translations.namespaces.'.$namespace)];
                }
            ),
            'locale' => $this->languageRepository->allLanguages(),
/**
 * 'group'     => CachedTranslation::pluck('group')->unique()->mapWithKeys(
 * function ($group) {
 * return [$group => __('translations.groups.' . $group)];
 * }
 * )->sort(),
 */ 'type' => [
                'lumen' => __('translations.types.lumen'),
                'inside' => __('translations.types.inside'),
                'modules' => __('translations.types.modules'),
                'back' => __('translations.types.back'),
                'front' => __('translations.types.front'),
                'database' => __('translations.types.database'),
                'self' => __('translations.types.self'),
            ],
            'locked' => [
                true => __('translations.locked.yes'),
                false => __('translations.locked.no'),
            ],
            'state' => [
                'translated' => __('translations.state.translated'),
                'untranslated' => __('translations.state.untranslated'),
            ],
            'draft' => [
                true => __('translations.status.draft'),
                false => __('translations.status.published'),
            ],
        ];
    }

    /**
     * Index all translations filtered by $filters
     *
     * @param  array  $filters
     * @return \Illuminate\Contracts\Pagination\LengthAwarePaginator|Collection
     * @throws Exception
     */
    public function indexTranslations(array $filters)
    {
        $locale = $this->getLocaleFromFilters($filters);
        if ($locale === null) {
            $locale = App::getLocale();
        }
        Translation::loadCachedTranslations();
        $allowedFilters = ['namespace', 'group', 'type'];

        /**
         * @var \Inside\Database\Eloquent\Builder $query
         */
        $query = CachedTranslation::in($locale)->when(
            array_key_exists('state', $filters),
            function ($query) use ($filters) {
                if ($filters['state'] == 'untranslated') {
                    return $query->whereTranslated(false);
                } elseif ($filters['state'] == 'translated') {
                    return $query->whereTranslated(true);
                }

                return $query;
            }
        );

        foreach ($allowedFilters as $filter) {
            if (array_key_exists($filter, $filters) && ! empty($filters[$filter])) {
                $query->where($filter, $filters[$filter]);
            }
        }
        if (array_key_exists('query', $filters) && ! empty($filters['query'])) {
            $query->where(
                function ($query) use ($filters) {
                    $query->where('key', 'like', '%'.$filters['query'].'%')->orWhere(
                        'text',
                        'like',
                        '%'.$filters['query'].'%'
                    )->orWhereNull('text');
                }
            );
        }

        $limit = null;
        $result = collect();
        $paginate = array_key_exists('paginate', $filters) && (bool) $filters['paginate'];
        if (array_key_exists('limit', $filters) && (int) $filters['limit'] > 0) {
            $limit = $filters['limit'];
        }
        $query->select(['type', 'namespace', 'group', 'key', 'text', 'original', 'translated']);
        $query->addSelect(
            [
                'placeholder' => CachedTranslation::from('inside_cached_translations as cached')->select(
                    'cached.text as placeholder'
                )->where('cached.language_id', $this->defaultLocale->id)->whereColumn(
                    'inside_cached_translations.namespace',
                    'cached.namespace'
                )->whereColumn(
                    'inside_cached_translations.group',
                    'cached.group'
                )->whereColumn('inside_cached_translations.key', 'cached.key')->take(1),
                'exportable' => TranslationModel::selectRaw('count(*)')->where(
                    function ($query) {
                        $query->whereColumn(
                            'inside_cached_translations.key',
                            DB::raw(
                                "CONCAT(inside_translations.namespace,'::',inside_translations.group,'.',inside_translations.key)"
                            )
                        )->orWhereColumn(
                            'inside_cached_translations.key',
                            DB::raw("CONCAT(inside_translations.group,'.',inside_translations.key)")
                        );
                    }
                )->whereColumn('inside_cached_translations.language_id', 'inside_translations.language_id')->where(
                    'inside_translations.exportable',
                    true
                )->take(1),
            ]
        );

        $query = DB::table(DB::raw("({$query->toBase()->toSql()}) as sub"))
            ->mergeBindings($query->toBase());

        if (! empty($filters['query'])) {
            $query->where(function ($q) use ($filters) {
                $q->whereLike('placeholder', '%'.$filters['query'].'%')
                    ->orWhereLike('key', '%'.$filters['query'].'%');
            });
        }

        $sort = $filters['sort'] ?? 'key:asc';
        if (isset($sort)) {
            $sortArray = explode(':', $sort, 2);
            if (in_array($sortArray[0], ['key', 'text'])) {
                $by = isset($sortArray[1]) ? $sortArray[1] : 'asc';
                $query->orderBy($sortArray[0], $by);
            }
        }

        if (! $paginate) {
            if (array_key_exists('offset', $filters) && (int) $filters['offset'] > 0) {
                $query->skip((int) $filters['offset']);
            }
            if ($limit !== null) {
                $query->limit($limit);
            }

            $result['data'] = $query->get();
        } else {
            if ($limit === null) {
                $limit = 10; // Default limit
            }
            $result = ContentHelper::getComplexePagination($query, $limit);
        }

        return $result;
    }

    /**
     * Get all translations from any source filtered by $filters
     *
     * @param array $filters
     * @return Collection
     */
    public function getAllTranslations(array $filters): Collection
    {
        $translations = collect();

        if (! isset($filters['locale'])) {
            $filters['locale'] = App::getLocale();
        }

        // Get core translations
        $translations = $translations->merge($this->getCoreTranslations($filters));

        // Get front translations
        $translations = $translations->merge($this->getFrontTranslations($filters));

        // Get database translations
        $translations = $translations->merge($this->getDatabaseTranslations($filters));

        $allowedFilters = $this->prepareAllowedFilters($translations);

        $translations = $this->filterTranslations($filters, $translations);

        return $this->prepareTranslationsForResponse($filters, $translations, $allowedFilters);
    }

    /**
     * Get all translations from any source using correct overloading order
     *
     * @param string $locale
     * @return Collection
     */
    public function getAllTranslationsFromAnySource(string $locale): Collection
    {
        $translations = collect();

        // TODO: switch back order when front will remove all english

        // Get front translations
        $translations = $translations->merge($this->getFrontTranslations(['locale' => $locale]));

        // Get core translations
        $translations = $translations->merge($this->getCoreTranslations(['locale' => $locale]));

        // Get database translations
        $translations = $translations->merge($this->getDatabaseTranslations(['locale' => $locale]));

        return $translations;
    }

    /**
     * get Front json from cache with default language fallback
     *
     * @return array
     * @throws Exception
     */
    public function getFastFrontFinalWithFallbacksTranslations(): array
    {
        $translations = [];
        $masterCachedTranslations = [];
        foreach ($this->languageRepository->allLanguages() as $locale => $languageName) {
            $result = [];
            $cachedTranslations = CachedTranslation::in($locale)->where('namespace', 'front')->pluck(
                'text',
                'key'
            );
            if ($locale == config('app.locale')) {
                $masterCachedTranslations = $cachedTranslations;
            }
            foreach ($cachedTranslations as $key => $text) {
                $fullKey = $key;
                $key = $this->getFrontKey($key);
                $keys = explode('.', $key);
                $cursor = &$result;
                foreach ($keys as $k) {
                    if (! is_array($cursor)) {
                        Log::warning('Front key conflict ['.$fullKey.']');
                        continue 2;
                    }
                    if (! isset($cursor[$k])) {
                        $cursor[$k] = [];
                    }
                    $cursor = &$cursor[$k];
                }
                $cursor = $text !== null ? $text : ($masterCachedTranslations[$fullKey] ?? null);
                if ($cursor !== null) {
                    $cursor = htmlspecialchars_decode($cursor);
                }
            }
            $translations[$locale] = $result;
        }

        return $translations;
    }

    /**
     * Load front final translations ( from our json )
     *
     * @return array
     */
    public function getFrontFinalTranslations(): array
    {
        $translations = [];
        foreach ($this->languageRepository->allLanguages() as $locale => $languageName) {
            $filters = [
                'namespace' => 'front',
                'draft' => false,
                'locale' => $locale,
            ];
            $localeTranslations = collect();

            // Get core translations
            $localeTranslations = $localeTranslations->merge($this->getCoreTranslations($filters));

            // Get front translations
            $localeTranslations = $localeTranslations->merge($this->getFrontTranslations($filters));

            // Get database translations
            $localeTranslations = $localeTranslations->merge($this->getDatabaseTranslations($filters));
            $result = [];
            foreach ($this->filterTranslations($filters, $localeTranslations) as $key => $translation) {
                $namespace = '*';
                if (strpos($key, '::') !== false) {
                    [$namespace, $key] = explode('::', $key, 2);
                }
                if ($namespace != 'front') {
                    continue;
                }
                $keys = explode('.', $key);
                $cursor = &$result;
                foreach ($keys as $k) {
                    if (! isset($cursor[$k])) {
                        $cursor[$k] = [];
                    }
                    $cursor = &$cursor[$k];
                }
                $cursor = $translation->text;
            }
            $translations[$locale] = $result;
        }

        return $translations;
    }

    /**
     * Extract current local from filters
     *
     * @param  array  $filters
     * @return string|null
     */
    protected function getLocaleFromFilters(array $filters): ?string
    {
        $locale = null;
        if (array_key_exists('locale', $filters) && ! empty($filters['locale'])) {
            $locale = $filters['locale'];
        }

        if ($locale !== null && ! $this->languageRepository->languageExists($locale)) {
            LanguageDoesNotExistsException::named($locale);
        }

        return $locale;
    }

    /**
     * Filter translations depending on front filters
     *
     * @param array $filters
     * @param Collection $translations
     * @return Collection
     */
    protected function filterTranslations(array $filters, Collection $translations): Collection
    {
        $allowedFilters = ['namespace', 'group', 'type', 'locked', 'draft'];
        $locale = $this->getLocaleFromFilters($filters);

        if ($locale === null) {
            $translations = $translations->whereIn('locale', $this->languageRepository->allLanguages()->keys());
        } else {
            $translations = $translations->where('locale', $locale);
        }

        if (array_key_exists('state', $filters) && $filters['state'] == 'untranslated') {
            $translations = $translations->filter(
                function ($translation, $key) {
                    return $translation->text === null || $translation->text === $key;
                }
            );
        } else {
            $translations = $translations->whereNotNull('text');
        }

        foreach ($allowedFilters as $filter) {
            if (array_key_exists($filter, $filters) && ! empty($filters[$filter])) {
                $translations = $translations->where($filter, $filters[$filter]);
            }
        }
        if (array_key_exists('query', $filters) && ! empty($filters['query'])) {
            $translations = $translations->filter(
                function ($translation, $key) use ($filters) {
                    return $translation->text !== null
                        && (false !== stristr($key, $filters['query'])
                            || false !== stristr($translation->text, $filters['query'])
                            || false !== stristr(
                                Lang::get($key, [], $this->defaultLocale->locale),
                                $filters['query']
                            ));
                }
            );
        }

        return $translations;
    }

    /**
     * Get translations from database ( overrides )
     *
     * @param array $filters
     * @return Collection
     */
    protected function getDatabaseTranslations(array $filters): Collection
    {
        $locale = null;
        if (array_key_exists('locale', $filters) && ! empty($filters['locale'])) {
            $locale = $filters['locale'];
        }

        if ($locale !== null && ! $this->languageRepository->languageExists($locale)) {
            LanguageDoesNotExistsException::named($locale);
        }
        $languages = $this->languageRepository->allLanguageIds()->flip();

        return collect(
            $this->getModel()->when(
                $locale !== null,
                function ($query) use ($locale) {
                    $query->whereHas(
                        'language',
                        function ($subQuery) use ($locale) {
                            $subQuery->where('locale', $locale);
                        }
                    );
                }
            )->whereNotNull('key')->get()->mapWithKeys(
                function ($translation) use ($languages) {
                    return [
                        $this->getFullTranslationKey(
                            $translation->namespace,
                            $translation->group,
                            $translation->key
                        ) => (object) [
                            'id' => $translation->id,
                            'locale' => $languages[$translation->language_id] ?? config('app.locale'),
                            'group' => $translation->group,
                            'namespace' => $translation->namespace,
                            'type' => 'database',
                            'key' => $translation->key,
                            'draft' => $translation->draft,
                            'locked' => $translation->locked,
                            'text' => $translation->text,
                        ],
                    ];
                }
            )
        );
    }

    /**
     * Get core translations ( lumen, inside, modules )
     *
     * @param array $filters
     * @return Collection
     */
    public function getCoreTranslations(array $filters): Collection
    {
        $namespace = null;
        $translations = collect();
        if (array_key_exists('namespace', $filters)) {
            $namespace = $filters['namespace'];
        }
        $locale = $this->getLocaleFromFilters($filters);

        if ($namespace === null || $namespace === '*') {
            // Get lumen translations
            $translations = $translations->merge(
                $this->getArrayTranslations('vendor/maecia-fork/lumen-framework/resources/lang', 'lumen', $locale)
            );

            // Get core back translations
            foreach (File::directories('vendor/maecia/inside') as $directory) {
                if (File::isDirectory($directory.'/resources/lang')) {
                    $translations = $translations
                        ->merge($this->getArrayTranslations($directory.'/resources/lang', 'inside', $locale))
                        ->merge($this->getJsonTranslations($directory.'/resources/lang', $locale));
                }
            }

            // get Inside automatics translations
            $translations = $translations->merge(
                $this->getJsonTranslations('vendor/maecia/inside/lang/automatics', $locale)
            );

            // get Inside custom translations
            $translations = $translations->merge(
                $this->getJsonTranslations('vendor/maecia/inside/core/resources/custom/lang', $locale)
            );

            // Get modules translations
            foreach (Package::list() as $packageName => $information) {
                if ($information->getType() !== 'inside' || ! $information->hasLanguage() || $information->getName() === 'maecia/inside') {
                    continue;
                }

                $translations = $translations->merge(
                    $this->getJsonTranslations('vendor/'.$information->getName().'/resources/lang', $locale)
                );
            }
        }

        return $translations;
    }

    /**
     * get Front translations from extractions
     *
     * @param array $filters
     * @return Collection
     */
    public function getFrontTranslations(array $filters): Collection
    {
        $translations = collect();
        $translationDirectories = [
            cms_base_path('vendor/maecia/inside/i18n/front/lang'),
            cms_base_path('vendor/maecia/inside/lang'),
        ];

        $locale = $this->getLocaleFromFilters($filters);
        if ($locale === null) {
            $locale = App::getLocale();
        }

        $finder = Finder::create()->files()->ignoreDotFiles(true)->in($translationDirectories)->name(
            $locale.'.json'
        );
        foreach ($finder->files() as $file) {
            if (Str::endsWith($file->getPath(), 'automatics') || Str::endsWith($file->getPath(), 'manifests')) {
                continue;
            }
            $currentLocale = $file->getBasename('.'.$file->getExtension());

            $decoded = collect(Arr::dot(json_decode($file->getContents(), true)));

            $translations = $translations->merge(
                $decoded->mapWithKeys(
                    function ($value, $key) use ($currentLocale) {
                        $group = '';
                        if (strpos($key, '.')) {
                            [$group, $key] = explode('.', $key, 2);
                        }

                        return [
                            $this->getFullTranslationKey('front', $group, $key) => (object) [
                                'locale' => $currentLocale,
                                'namespace' => 'front',
                                'type' => 'front',
                                'group' => $group,
                                'key' => $key,
                                'text' => is_string($value) ? $value : null,
                                'draft' => false,
                                'locked' => false,
                            ],
                        ];
                    }
                )
            );
        }

        return $translations;
    }

    /**
     * get Translations from php arrays
     *
     * @param string $path
     * @param string $type
     * @param string|null $locale
     * @return Collection
     */
    public function getArrayTranslations(string $path, string $type, ?string $locale = null): Collection
    {
        $translations = collect();
        $translationDirectory = cms_base_path($path);
        $finder =
            Finder::create()->files()->ignoreDotFiles(true)->in($translationDirectory)->exclude('storage')->exclude(
                'vendor'
            )->name('*.php');

        foreach ($finder->files() as $file) {
            $group = $file->getBasename('.'.$file->getExtension());
            $explodedPath =
                explode(DIRECTORY_SEPARATOR, trim(str_replace($translationDirectory, '', $file->getRealPath())));

            if (count($explodedPath) < 2) {
                continue;
            }

            $currentLocale = $explodedPath[count($explodedPath) - 2];
            if ($locale !== null && $currentLocale !== $locale) {
                continue;
            }

            $localTranslation = collect(Arr::dot(include $file->getRealPath()));
            $localTranslation = $localTranslation->filter(
                function ($translation) {
                    return is_string($translation);
                }
            );
            $translations = $translations->merge(
                $localTranslation->mapWithKeys(
                    function ($value, $key) use ($currentLocale, $group, $type) {
                        return [
                            $this->getFullTranslationKey('*', $group, $key) => (object) [
                                'locale' => $currentLocale,
                                'namespace' => '*',
                                'type' => $type,
                                'group' => $group,
                                'key' => $key,
                                'text' => $value,
                                'draft' => false,
                                'locked' => false,
                            ],
                        ];
                    }
                )
            );
        }

        return $translations;
    }

    /**
     * Get back json translations
     *
     * @param string $path
     * @param string|null $locale
     * @return Collection
     */
    public function getJsonTranslations(string $path, ?string $locale = null): Collection
    {
        $translations = collect();
        $translationDirectory = cms_base_path($path);
        $finder =
            Finder::create()->files()->ignoreDotFiles(true)->in($translationDirectory)->exclude('storage')->exclude(
                'vendor'
            )->name($locale === null ? '*.json' : ($locale.'.json'));
        foreach ($finder->files() as $file) {
            $locale = $file->getBasename('.'.$file->getExtension());

            $decoded = collect(json_decode($file->getContents(), true));

            $isBack = (1 === preg_match(
                '#'.config('app.code').'-back/resources/lang/'.$file->getBasename().'$#',
                $file->getRealPath()
            ));

            $translations = $translations->merge(
                $decoded->mapWithKeys(
                    function ($value, $key) use ($locale, $isBack) {
                        $group = 'self';
                        if (Str::startsWith($key, 'front::')) {
                            // This is a front translation in a back package (mostly autmatics)
                            $key = Str::after($key, 'front::');
                            if (strpos($key, '.')) {
                                [$group, $key] = explode('.', $key, 2);
                            }
                            if (Str::startsWith($key, 'front::')) {
                                return;
                            }

                            return [
                                $this->getFullTranslationKey('front', $group, $key) => (object) [
                                    'locale' => $locale,
                                    'namespace' => 'front',
                                    'type' => 'front',
                                    'group' => $group,
                                    'key' => $key,
                                    'text' => is_string($value) ? $value : null,
                                    'draft' => false,
                                    'locked' => false,
                                ],
                            ];
                        }
                        if (strpos($key, '.')) {
                            [$group, $key] = explode('.', $key, 2);
                        }

                        return [
                            $this->getFullTranslationKey('*', $group, $key) => (object) [
                                'locale' => $locale,
                                'namespace' => '*',
                                'type' => $isBack ? 'back' : 'modules',
                                'group' => $group,
                                'key' => $group.'.'.$key,
                                'text' => $value,
                                'draft' => false,
                                'locked' => false,
                            ],
                        ];
                    }
                )
            );
        }

        return $translations;
    }

    /**
     * get full translation key from translation fields
     *
     * @param  string  $namespace
     * @param  string  $group
     * @param  string  $key
     * @return string
     */
    protected function getFullTranslationKey(string $namespace, string $group, string $key): string
    {
        return $namespace === '*' ? "{$group}.{$key}" : "{$namespace}::{$group}.{$key}";
    }

    /**
     * Scan php files for usage of translations
     *
     * @return array[]
     */
    public function findTranslations()
    {
        $results = ['single' => [], 'group' => []];
        $methods = ['__', 'trans', 'trans_choice', '$_tr', '$_tc', 'Lang::getFromJson'];
        $disk = new Filesystem();

        $scanPaths = [];
        foreach (Package::list() as $packageName => $information) {
            if ($information->getType() != 'inside') {
                continue;
            }
            $scanPaths[] = cms_base_path('vendor/'.$information->getName());
        }
        // Add front
        $scanPaths[] = cms_base_path('front');

        $groupPattern =                          // See https://regex101.com/r/WEJqdL/6
            "[^\w|>]".                          // Must not have an alphanum or _ or > before real method
            '('.implode('|', $methods).')'.  // Must start with one of the functions
            "\(".                               // Match opening parenthesis
            "[\'\"]".                           // Match " or '
            '('.                                // Start a new group to match:
            '[a-zA-Z0-9_-]+'.                   // Must start with group
            "([.](?! )[^\1)]+)+".               // Be followed by one or more items/keys
            ')'.                                // Close group
            "[\'\"]".                           // Closing quote
            "[\),]";                             // Close parentheses or new parameter

        $stringPattern = "[^\w]".                          // Must not have an alphanum before real method
            '('.implode('|', $methods).')'.       // Must start with one of the functions
            "\(\s*".                                       // Match opening parenthesis
            "(?P<quote>['\"])".                            // Match " or ' and store in {quote}
            "(?P<string>(?:\\\k{quote}|(?!\k{quote}).)*)". // Match any string that can be {quote} escaped
            "\k{quote}".                                   // Match " or ' previously matched
            "\s*[\),]";                                     // Close parentheses or new parameter

        $groupKeys = [];
        $stringKeys = [];

        $finder = Finder::create()->files()->ignoreDotFiles(true)->in($scanPaths)->exclude('storage')->exclude('vendor')
            ->name(
                '*.php'
            )->name('*.vue');

        foreach ($finder->files() as $file) {
            if (preg_match_all("/$groupPattern/siU", $file->getContents(), $matches)) {
                // Get all matches
                foreach ($matches[2] as $key) {
                    $groupKeys[] = $key;
                }
            }

            if (preg_match_all("/$stringPattern/siU", $file->getContents(), $matches)) {
                foreach ($matches['string'] as $key) {
                    if (preg_match("/(^[a-zA-Z0-9_-]+([.][^\1) ]+)+$)/siU", $key, $groupMatches)) {
                        continue;
                    }
                    if (! (Str::contains($key, '::') && Str::contains($key, '.'))
                        || Str::contains($key, ' ')
                    ) {
                        $stringKeys[] = $key;
                    }
                }
            }
        }
        $groupKeys = array_unique($groupKeys);
        $stringKeys = array_unique($stringKeys);
        //  dd($groupKeys, $stringKeys);

        // TODO

        return $results;
    }

    /**
     * prepare allowed filters
     *
     * @param Collection $translations
     * @return array
     */
    protected function prepareAllowedFilters(Collection $translations): array
    {
        return [
            'namespace' => $translations->pluck('namespace')->unique()->mapWithKeys(
                function ($namespace) {
                    return [$namespace => __('translations.namespaces.'.$namespace)];
                }
            ),
            'locale' => $this->languageRepository->allLanguages(),
            'group' => $translations->pluck('group')->unique()->mapWithKeys(
                function ($group) {
                    return [$group => __('translations.groups.'.$group)];
                }
            ),
            'type' => [
                'lumen' => __('translations.types.lumen'),
                'inside' => __('translations.types.inside'),
                'modules' => __('translations.types.modules'),
                'back' => __('translations.types.back'),
                'front' => __('translations.types.front'),
                'database' => __('translations.types.database'),
                'self' => __('translations.types.self'),
            ],
            'locked' => [
                true => __('translations.locked.yes'),
                false => __('translations.locked.no'),
            ],
            'state' => [
                'translated' => __('translations.state.translated'),
                'untranslated' => __('translations.state.untranslated'),
            ],
            'draft' => [
                true => __('translations.status.draft'),
                false => __('translations.status.published'),
            ],
        ];
    }

    /**
     * prepare translations for json response
     *
     * @param array $filters
     * @param Collection $translations
     * @param array $availableFilters
     * @return Collection
     */
    protected function prepareTranslationsForResponse(array $filters, Collection $translations, array $availableFilters)
    {
        $result = collect(
            [
                'filters' => $availableFilters,
            ]
        );

        // SortKeys
        $translations = $translations->toArray();
        ksort($translations);
        $translations = collect($translations);
        if (array_key_exists('sort', $filters)
            && in_array(
                $filters['sort'],
                ['namespace', 'locale', 'group', 'type', 'locked', 'draft']
            )
        ) {
            $translations = $translations->sortBy($filters['sort']);
        }

        $limit = null;
        $paginate = array_key_exists('paginate', $filters) && (bool) $filters['paginate'];
        if (array_key_exists('limit', $filters) && (int) $filters['limit'] > 0) {
            $limit = (int) $filters['limit'];
        }

        if (! $paginate) {
            if (array_key_exists('offset', $filters) && (int) $filters['offset'] > 0) {
                $translations = $translations->skip((int) $filters['offset']);
            }
            if ($limit !== null) {
                $translations = $translations->take($limit);
            }

            $result['data'] = $translations;
        } else {
            if ($limit === null) {
                $limit = 10; // Default limit
            }
            // Prepare pagination
            $currentPage = LengthAwarePaginator::resolveCurrentPage();
            $currentPageItems = $translations->slice(($currentPage - 1) * $limit, $limit);

            $result = $result->merge(
                new LengthAwarePaginator(
                    $currentPageItems,
                    $translations->count(),
                    $limit,
                    $currentPage
                )
            );
        }

        return $result;
    }
}
