<?php

declare(strict_types=1);

namespace Inside\I18n\Repositories;

use Illuminate\Support\Collection;
use Inside\Application;
use Inside\I18n\Exceptions\LanguageDoesNotExistsException;
use Inside\I18n\Models\Language;

final class LanguageRepository extends AbstractRepository
{
    protected Application $app;

    public function __construct(Language $model, Application $app)
    {
        $this->model = $model;

        $this->app = $app;
    }

    /**
     * Get a language model
     *
     * @param string $locale
     * @return Language
     */
    public function getLanguage(string $locale): Language
    {
        if (! $this->languageExists($locale)) {
            LanguageDoesNotExistsException::named($locale);
        }

        return Language::where('locale', $locale)->first();
    }

    /**
     * Does language exist and is supported by current instance
     *
     * @param string $locale
     * @return bool
     */
    public function languageExists(string $locale): bool
    {
        return Language::where('locale', $locale)->exists() && false !== array_search($locale, list_languages());
    }

    /**
     * Get All supported languages for current instance
     *
     * @return Collection
     */
    public function allLanguages(): Collection
    {
        return Language::whereIn('locale', list_languages())->get()->sortBy(function ($locale) {
            return $locale == config('app.locale', 'fr');
        })->mapWithKeys(
            function ($language) {
                return [$language->locale => $language->name];
            }
        );
    }

    public function allKnownLanguages(): Collection
    {
        return Language::get()->sortBy(function ($locale) {
            return $locale == config('app.locale', 'fr');
        })->mapWithKeys(
            function ($language) {
                return [$language->locale => $language->name];
            }
        );
    }

    /**
     * Get All supported languages for current instance with ids
     *
     * @return Collection
     */
    public function allLanguageIds(): Collection
    {
        return Language::whereIn('locale', list_languages())->get()->sortBy(function ($locale) {
            return $locale == config('app.locale', 'fr');
        })->mapWithKeys(
            function ($language) {
                return [$language->locale => $language->id];
            }
        );
    }
}
