<?php

declare(strict_types=1);

namespace Inside\I18n\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Inside\Database\Eloquent\WithEnhancedBuilder;

/**
 * Class CachedTranslation
 *
 * @property int $id
 * @property int $language_id
 * @property string $type
 * @property string $namespace
 * @property string $group
 * @property string $key
 * @property string|null $text
 * @property bool $translated
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property-read Translation $core
 * @property-read mixed $locale
 * @property-read Language $language
 * @method static Builder|CachedTranslation in($locale)
 * @method static Builder|CachedTranslation keyed($key)
 * @method static \Inside\Database\Eloquent\Builder|CachedTranslation newModelQuery()
 * @method static \Inside\Database\Eloquent\Builder|CachedTranslation newQuery()
 * @method static \Inside\Database\Eloquent\Builder|CachedTranslation query()
 * @method static Builder|CachedTranslation whereCreatedAt($value)
 * @method static Builder|CachedTranslation whereGroup($value)
 * @method static Builder|CachedTranslation whereId($value)
 * @method static Builder|CachedTranslation whereKey($value)
 * @method static Builder|CachedTranslation whereLanguageId($value)
 * @method static Builder|CachedTranslation whereNamespace($value)
 * @method static Builder|CachedTranslation whereText($value)
 * @method static Builder|CachedTranslation whereTranslated($value)
 * @method static Builder|CachedTranslation whereType($value)
 * @method static Builder|CachedTranslation whereUpdatedAt($value)
 *
 * @mixin Builder
 */
final class CachedTranslation extends Model
{
    use WithEnhancedBuilder;

    protected $table = 'inside_cached_translations';

    protected $fillable = ['language_id', 'namespace', 'group', 'key', 'text', 'type', 'translated'];

    protected $casts = [
        'translated' => 'boolean',
    ];

    public function language(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Language::class);
    }

    public function core(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Translation::class, 'key', 'key');
    }

    public function getLocaleAttribute(): string
    {
        return $this->language->locale;
    }

    /**
     * scope locale
     *
     * @param Builder $query
     * @param string $locale
     * @return Builder
     */
    public function scopeIn(Builder $query, string $locale): Builder
    {
        return $query->whereHas(
            'language',
            function ($query) use ($locale) {
                $query->where('locale', $locale);
            }
        );
    }

    /**
     * search by full key
     *
     * @param Builder $query
     * @param string $key
     * @return Builder
     */
    public function scopeKeyed(Builder $query, string $key): Builder
    {
        $namespace = null;
        $group = null;
        if (str_contains($key, '::')) {
            [$namespace, $shortKey] = explode('::', $key, 2);
        } else {
            $shortKey = $key;
        }
        if (str_contains($shortKey, '.')) {
            [$group, $unused] = explode('.', $shortKey, 2);
        }

        return $query->when(
            $group !== null,
            function ($query) use ($group) {
                $query->where('group', $group);
            }
        )->when(
            $namespace !== null,
            function ($query) use ($namespace) {
                $query->where('namespace', $namespace);
            }
        )->where('key', $key);
    }
}
