<?php

namespace Inside\I18n\Console\Commands;

use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Inside\Console\Command;
use Inside\Facades\Http;
use Inside\I18n\Facades\Translation;
use Inside\I18n\Jobs\ClearCachedTranslations;
use Inside\I18n\Jobs\SynchronizeFrontTranslations;
use Inside\I18n\Repositories\TranslationRepository;
use Inside\Support\Str;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class AddLanguageCommand extends Command
{
    /** @var TranslationRepository */
    protected $translationRepository;

    protected $signature = 'inside:translation:add {locale}';

    public function __construct(TranslationRepository $translationRepository)
    {
        parent::__construct();
        $this->translationRepository = $translationRepository;
    }

    /**
     * @throws FileNotFoundException
     */
    public function handle(): void
    {
        // Load front
        Lang::addJsonPath(cms_base_path('vendor/maecia/inside/i18n/front/lang'));
        Lang::addJsonPath(cms_base_path('vendor/maecia/inside/lang'));

        $this->write('Synchronisation des traductions du front');
        SynchronizeFrontTranslations::dispatchNow();
        $this->writeResult(true);
        $this->write('Suppression du cache de traduction');
        ClearCachedTranslations::dispatchNow();
        $this->writeResult(true);
        $this->write('Reconstruction du cache de traduction');
        Translation::loadCachedTranslations();
        $this->writeResult(true);
        // Check lang folder
        /** @var string $locale */
        $locale = $this->argument('locale');
        $this->info("Ajout de la langue <fg=cyan>$locale</>");
        $repository = cms_base_path('vendor/maecia/inside/lang');
        $branch = 'lang/'.now()->format('Ymdhis').'_'.$locale;

        try {
            $this->write("Préparation du repo et création de la branche <fg=yellow>$branch</>");
            $cmd = "cd $repository && git fetch origin && git reset --hard origin/master && git pull origin master && git checkout -b $branch";
            $process = Process::fromShellCommandline($cmd, null, null, null, 300);
            $process->run();
            if (! $process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }
        } catch (\Exception $e) {
            $this->writeResult(false);
            throw new ProcessFailedException($process);
        }
        $this->writeResult(true);
        $this->write("Fichier de traduction automatique de la langue <fg=cyan>$locale</> existant ?");
        $baseFileExists = false;
        $automaticTranslations = collect();
        $baseLangFile = $repository.'/automatics/'.$locale.'.json';
        if (File::exists($baseLangFile)) {
            $this->writeResult(true);
            $this->write('Chargement des langues existantes ');
            $automaticTranslations = collect(json_decode(File::get($baseLangFile), true));
            $baseFileExists = true;
            $this->writeResult(true);
        } else {
            $this->writeResult(false);
        }
        $this->write("Fichier de manifest de traduction de la langue <fg=cyan>$locale</> existant ?");
        $manifestFileExists = false;
        $manifestTranslations = collect();
        $manifestLangFile = $repository.'/manifests/'.$locale.'.json';
        if (File::exists($manifestLangFile)) {
            $this->writeResult(true);
            $this->write('Chargement des manifestes de traduction existantes ');
            $manifestTranslations = collect(json_decode(File::get($manifestLangFile), true));
            $manifestFileExists = true;
            $this->writeResult(true);
        } else {
            $this->writeResult(false);
        }
        $this->info('Début de la récupération des traductions ..');
        $this->line(str_repeat('_', 80));
        foreach ($this->translationRepository->getAllKnownKeys() as $key => $translationInfo) {
            if ($automaticTranslations->has($key)) {
                // Already got from a previous auto load
                continue;
            }
            /** @var string $frenchTranslation */
            $frenchTranslation = __($key, [], 'fr');
            if ($frenchTranslation == $key) {
                $this->warning('Aucune traduction française pour la clé <fg=yellow>'.$key.'</>');
                continue; // French does not exists !
            }
            $localeTranslation = __($key, [], $locale);
            if ($localeTranslation != $frenchTranslation) {
                // Already in the system
                continue;
            }
            $this->write('Traduction (<fg=yellow>'.$key.'</>) de <fg=cyan>'.$frenchTranslation.'</> en <fg=green>'.$locale.'</>');
            $translation = $this->translate($frenchTranslation, $locale);
            if ($translation === null) {
                $this->writeResult(false);
                continue;
            }
            $this->write(' => <fg=magenta>'.$translation.'</>');
            $this->writeResult(true);

            $automaticTranslations[$key] = $translation;
            $manifestTranslations[$key] = [
                'key' => $key,
                'type' => $translationInfo['type'],
                'namespace' => $translationInfo['namespace'],
                'groupe' => $translationInfo['group'],
                'text' => $translation,
            ];
        }
        $this->write("Sauvegarde du fichier <fg=green>$baseLangFile</> au repo");
        $this->writeResult(false != File::put($baseLangFile, $automaticTranslations->toJson(JSON_PRETTY_PRINT)));
        if (! $baseFileExists) {
            try {
                $this->write("Ajouter le fichier de traduction automatique <fg=green>$baseLangFile</> au repo");
                $cmd = "cd $repository && git add automatics/$locale.json";
                $process = Process::fromShellCommandline($cmd, null, null, null, 300);
                $process->run();
                if (! $process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            } catch (\Exception $e) {
                $this->writeResult(false);
                throw new ProcessFailedException($process);
            }
        }
        $this->writeResult($process->isSuccessful());
        $this->write("Sauvegarde du fichier manifest <fg=green>$manifestLangFile</> au repo");
        $this->writeResult(false != File::put($manifestLangFile, $manifestTranslations->toJson(JSON_PRETTY_PRINT)));
        if (! $manifestFileExists) {
            try {
                $this->write("Ajouter le fichier de traduction manifest <fg=green>$manifestLangFile</> au repo");
                $cmd = "cd $repository && git add manifests/$locale.json";
                $process = Process::fromShellCommandline($cmd, null, null, null, 300);
                $process->run();
                if (! $process->isSuccessful()) {
                    throw new ProcessFailedException($process);
                }
            } catch (\Exception $e) {
                $this->writeResult(false);
                throw new ProcessFailedException($process);
            }
        }
        $this->writeResult($process->isSuccessful());

        try {
            $this->write("Commit des fichiers de langue automatique pour la langue <fg=yellow>$locale</>");
            $date = now()->format('m/d/Y H:i:s');
            $cmd = "cd $repository && git commit -a -m 'trad: Automatic translation from $date in $locale'";
            $process = Process::fromShellCommandline($cmd, null, null, null, 300);
            $process->run();
            if (! $process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }
        } catch (\Exception $e) {
            $this->writeResult(false);
            throw new ProcessFailedException($process);
        }
        $this->writeResult($process->isSuccessful());

        try {
            $this->write('Téléversement sur Bitbucket');
            $cmd = "cd $repository && git push --set-upstream origin $branch";
            $process = Process::fromShellCommandline($cmd, null, null, null, 300);
            $process->run();
            if (! $process->isSuccessful()) {
                throw new ProcessFailedException($process);
            }
        } catch (\Exception $e) {
            $this->writeResult(false);
            throw new ProcessFailedException($process);
        }
        $this->writeResult($process->isSuccessful());
        $this->info('Création des traductions terminées : Il ne reste plus qu\'a vérifier la branche');
        $this->write('Suppression du cache de traduction pour prendre en compte les nouvelles traductions');
        ClearCachedTranslations::dispatchNow();
        $this->call('cache:clear');
    }

    /**
     * @param string $inFrench
     * @param string $locale
     * @return mixed|string|void|null
     */
    protected function translate(string $inFrench, string $locale)
    {
        $count = Cache::store('file')->get('inside.deepl.count', 0) + strlen($inFrench);
        Cache::store('file')->forever('inside.deepl.count', $count);
        if ($count > 500000) {
            $this->error('Quota atteint, revenez plus tard !');
            die();
        }
        $response = Http::get(config('translation.deepl.uri'), [
            'auth_key' => config('translation.deepl.auth_key'),
            'text' => $inFrench,
            'source_lang' => 'FR',
            'target_lang' => Str::upper($locale),
        ]);
        if ($response->successful()) {
            $jsonReponse = $response->json();
            $firstTranslation = Arr::first($jsonReponse['translations']);
            if ($firstTranslation) {
                return $firstTranslation['text'];
            }
        }

        return null;
    }
}
