<?php

namespace Inside\Documentation\Services;

use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Yaml\Dumper as YamlDumper;

/**
 * Class Generator
 *
 * @package Inside\Documentation\Services
 */
class Generator
{
    /**
     * @var \OpenApi\Annotations\OpenApi
     */
    protected $swagger;

    /**
     * @var string
     */
    protected $docDir;

    /**
     * @var string
     */
    protected $docsFile;

    /**
     * @var string
     */
    protected $yamlDocsFile;

    /**
     * @var array
     */
    protected $excludedDirs;

    /**
     * @var array
     */
    protected $constants;

    /**
     * Generator constructor.
     *
     */
    public function __construct()
    {
        $this->docDir       = config('documentation.paths.docs');
        $this->docsFile     = $this->docDir . '/' . config('documentation.docs_json', 'api-docs.json');
        $this->yamlDocsFile = $this->docDir . '/' . config('documentation.paths.docs_yaml', 'api-docs.yaml');
        $this->excludedDirs = config('documentation.paths.excludes');
        $this->constants    = config('documentation.constants') ?: [];
    }

    /**
     * Generate doc in docDir
     */
    public static function generateDocs(): void
    {
        (new self())->prepareDirectory()->defineConstants()->scanFilesForDocumentation()->populateServers()->saveToJson(
        )->convertToYaml();
    }

    /**
     * prepare docDir
     *
     * @return $this
     * @throws \Exception
     */
    protected function prepareDirectory(): self
    {
        if (File::exists($this->docDir) && !is_writable($this->docDir)) {
            throw new \Exception('Documentation storage directory is not writable');
        }
        if (File::exists($this->docDir)) {
            File::deleteDirectory($this->docDir);
        }
        File::makeDirectory($this->docDir);

        return $this;
    }

    /**
     * dedine constant to be used in annotations
     *
     * @return $this
     */
    protected function defineConstants(): self
    {
        if (!empty($this->constants)) {
            foreach ($this->constants as $key => $value) {
                defined($key) || define($key, $value);
            }
        }

        return $this;
    }

    /**
     * scan file for annotations
     *
     * @return $this
     */
    protected function scanFilesForDocumentation(): self
    {
        $finder = new Finder();

        $composers =
            $finder->in(cms_base_path('vendor/maecia/'))->files()->name('composer.json')->depth('<= 3')->followLinks();
        $toScans   = [];
        foreach ($composers as $file) {
            $jsonFile = new \Composer\Json\JsonFile($file->getRealPath());
            try {
                $composer = $jsonFile->read();

                if (isset($composer['autoload']) && isset($composer['autoload']['psr-4'])) {
                    $toScans[] = $file->getPath() . '/' . array_pop($composer['autoload']['psr-4']);
                }
            } catch (\Seld\JsonLint\ParsingException $e) {
                Log::error($e->getMessage(), ['exception' => $e]);
            }
        }

        $this->swagger = \OpenApi\Generator::scan(
            $toScans,
            ['exclude' => $this->excludedDirs]
        );

        return $this;
    }

    /**
     * Add serveur informations from config
     *
     * @return $this
     */
    protected function populateServers(): self
    {
        if (config('documentation.paths.base') !== null) {
            $this->swagger->servers = [
                new \OpenApi\Annotations\Server(['url' => config('documentation.paths.base')]),
            ];
        }

        return $this;
    }

    /**
     * Save swagger to a json file
     *
     * @return $this
     * @throws \Exception
     */
    protected function saveToJson(): self
    {
        $this->swagger->saveAs($this->docsFile);

        return $this;
    }

    /**
     * Convert json file to yaml
     *
     */
    protected function convertToYaml(): void
    {
        $contents = file_get_contents($this->docsFile);

        if ($contents === false) {
            return;
        }

        file_put_contents(
            $this->yamlDocsFile,
            (new YamlDumper(2))->dump(json_decode($contents, true), 20)
        );
    }
}
