<?php

namespace Inside\Documentation\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Services\Schema\SchemaService;
use Inside\Documentation\Facades\Cypress;
use Inside\Host\Bridge\BridgeContent;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Models\Role;
use Inside\Validation\ValidateRequests;

class CypressController
{
    use ValidateRequests;

    /**
     * get random correct content data
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     */
    public function factory(Request $request): JsonResponse
    {
        $data = $this->validateData($request->all(), [
            'type' => [
                'required',
                function ($attribute, $value, $fail) {
                    if (!Schema::hasContentType($value)) {
                        $fail(__('validations.invalid_content_type', ['type' => $value]));
                    }
                },
            ],
        ]);

        return response()->json(['data' => Cypress::factory($data['type'])]);
    }

    /**
     * Update a content type config
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     */
    public function updateContentTypeConfig(Request $request): JsonResponse
    {
        $data = $this->validateData($request->all(), [
            'type'    => [
                'required',
                function ($attribute, $value, $fail) {
                    if (!Schema::hasContentType($value)) {
                        $fail(__('validations.invalid_content_type', ['type' => $value]));
                    }
                },
            ],
            'options' => [
                'required',
                'array',
            ],
        ]);

        return response()->json([
            'status' => Cypress::updateContentTypeConfig($data['type'], $data['options']),
        ]);
    }

    /**
     * Update a field config
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     */
    public function updateFieldConfig(Request $request): JsonResponse
    {
        $data = $this->validateData($request->all(), [
            'type'      => [
                'required',
                function ($attribute, $value, $fail) {
                    if (!Schema::hasContentType($value)) {
                        $fail(__('validations.invalid_content_type', ['type' => $value]));
                    }
                },
            ],
            'fieldName' => [
                'required',
                function ($attribute, $value, $fail) use ($request) {
                    $type = $request->input('type');
                    if ($type !== null && !Schema::hasField($type, $value)) {
                        $fail(__('validations.invalid_content_type', ['type' => $type, 'fieldName' => $value]));
                    }
                },
            ],
            'options'   => [
                'required',
                'array',
            ],
        ]);

        return response()->json([
            'status' => Cypress::updateFieldConfig($data['type'], $data['fieldName'], $data['options']),
        ]);
    }

    /**
     * get an image
     *
     * @param  Request  $request
     * @return JsonResponse
     * @throws Exception
     */
    public function getImage(Request $request): JsonResponse
    {
        $data = $this->validateData($request->all(), [
            'extension' => [
                'required',
            ],
        ]);

        return response()->json(['path' => Cypress::getRandomImage($data['extension'])]);
    }

    /**
     * get a file
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function getFile(Request $request): JsonResponse
    {
        $data = $this->validateData($request->all(), [
            'extension' => [
                'required',
            ],
        ]);

        return response()->json(['path' => Cypress::getRandomFile($data['extension'])]);
    }

    /**
     * Start a cypress test session
     *
     * @param  Request  $request
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function start(Request $request): JsonResponse
    {
        if (config('mail') === null) {
            // Mailer not loaded yet
            app('mailer');
        }
        // Check email driver
        if (config('mail.port') != '1025') { // MailHog port
            throw new Exception('Mail driver seems not to be using MailHog. Check .env and mail configuration before using cypress tests');
        }

        $this->checkAndPrepareDefaultRoles();

        $data = $this->validateData($request->all(), [
            'role' => [
                'required',
                'exists:inside_roles,name',
            ],
        ]);
        $role = $data['role'];

        // Create a random user
        $bridge = new BridgeContent();

        $data = Cypress::factory('users');

        $userUuid = $bridge->contentInsert('users', $data);

        // Force status to be true or nothing is possible !
        $bridge->contentUpdate('users', [
            'uuid' => $userUuid,
            'status' => true,
        ]);

        // Set user to the correct role
        $user = User::findOrFail($userUuid);
        $user->assignRole($role);

        // Log the user !
        $token = Authentication::logAs($user, 'cypress');

        return response()->json([
            'apiToken' => $token->plainTextToken,
            'email'    => $data['email'],
            'password' => $data['password'],
        ]);
    }

    /**
     * End cypress session and remove temp user
     *
     * @return JsonResponse
     */
    public function cleanup(): JsonResponse
    {
        return response()->json(['toDelete' => Cypress::cleanUp(Auth::user())]);
    }

    /**
     * check and prepare roles if necessary
     *
     * @throws Exception
     */
    protected function checkAndPrepareDefaultRoles(): void
    {
        if (!Role::whereName('administrator')->exists()) {
            Log::warning('[CypressionController::checkAndPrepareDefaultRoles] For some reason *administrator* role does not exists, we will create it');

            $role = Role::create([
                'name' => 'administrator',
            ]);

            // Administrator can create everything !
            $permissibleTypes = Schema::getContentTypes(function ($model) {
                return $model['type'] === SchemaService::CONTENT_MODEL_TYPE
                    && isset($model['options'])
                    && isset($model['options']['permissible'])
                    && $model['options']['permissible'];
            });
            $schemas          = [];
            foreach ($permissibleTypes as $type) {
                foreach (['read', 'create', 'edit', 'delete'] as $action) {
                    $schemas[] = [
                        'authorizable_type' => type_to_class($type),
                        'action'            => $action,
                        'children'          => true,
                        'invert'            => false,
                    ];
                }
            }
            PermissionSchema::create($schemas, $role->id);
            Permission::buildPermissionForRole($role->id);
        }
    }
}
