<?php

namespace Inside\Documentation\Console\Commands;

use Composer\Autoload\ClassMapGenerator;
use Inside\Console\Command;
use Inside\Documentation\Services\DocumentationHelper;

class DocumentationCommand extends Command
{
    /**
     * The console command signature.
     *
     * @var string
     */
    protected $signature = 'inside:documentation {dir}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate documentation for models';

    /**
     * Execute the console command.
     *
     * @return void
     */
    public function handle(): void
    {
        /** @var string|null $dir */
        $dir = $this->argument('dir');
        if (!$dir) {
            $this->error('You did not provide any dir path');
            return;
        }
        $dir = cms_base_path("/$dir");
        if (file_exists($dir)) {
            $this->generateDoc(ClassMapGenerator::createMap($dir));
        }
        $this->info("DONE");
    }

    protected function generateDoc(array $dirs, int $tries = 1): void
    {
        if (empty($dirs) || $tries > 20) {
            return;
        }
        $toBeRetries = [];
        foreach ($dirs as $model => $path) {
            $this->write("Generating documentation for Model [" . $model . "] (" . $tries . ")");
            try {
                // Generate Doc
                $realpath = realpath($path);
                if (!$realpath) {
                    throw new \LogicException("Invalid path: $path");
                }

                $documentation = new DocumentationHelper();
                $documentation->generateDoc(
                    $model,
                    realpath($path)
                );
            } catch (\Throwable $e) {
                $this->writeResult(false);
                $toBeRetries[$model] = $path;
            }
            $this->writeResult(true);
        }
        $this->generateDoc($toBeRetries, $tries++);
    }
}
