<?php

namespace Inside\Support\View\Compilers;

use Illuminate\Support\Arr;
use Illuminate\View\Compilers\BladeCompiler as BaseBladeCompiler;
use Inside\Support\Str;

class BladeCompiler extends BaseBladeCompiler
{
    use Concerns\CompilesComponents;
    use Concerns\CompilesEchos;
    use Concerns\CompilesHelpers;
    use Concerns\CompilesJson;
    use Concerns\CompilesLayouts;
    use Concerns\CompilesRawPhp;
    use Concerns\CompilesInjections;
    use Concerns\CompilesTranslations;

    /**
     * All of the registered precompilers.
     *
     * @var array
     */
    protected $precompilers = [];

    /**
     * The array of class component aliases and their class names.
     *
     * @var array
     */
    protected $classComponentAliases = [];

    /**
     * The array of class component namespaces to autoload from.
     *
     * @var array
     */
    protected $classComponentNamespaces = [];

    /**
     * Indicates if component tags should be compiled.
     *
     * @var bool
     */
    protected $compilesComponentTags = true;

    protected $echoFormat = 'es(%s)';

    /**
     * The array of anonymous component namespaces to autoload from.
     *
     * @var array<string, string>
     */
    protected array $anonymousComponentNamespaces = [];

    /**
     * compile string $value
     *
     * @param  string  $value
     *
     * @return mixed|string|string[]
     */
    public function compileString($value)
    {
        [$this->footer, $result] = [[], ''];

        // First we will compile the Blade component tags. This is a precompile style
        // step which compiles the component Blade tags into @component directives
        // that may be used by Blade. Then we should call any other precompilers.
        $value = $this->compileComponentTags(
            $this->compileComments($this->storeUncompiledBlocks($value))
        );

        foreach ($this->precompilers as $precompiler) {
            $value = call_user_func($precompiler, $value);
        }

        // Here we will loop through all of the tokens returned by the Zend lexer and
        // parse each one into the corresponding valid PHP. We will then have this
        // template as the correctly rendered PHP that can be rendered natively.
        foreach (token_get_all($value) as $token) {
            $result .= is_array($token) ? $this->parseToken($token) : $token;
        }

        if (! empty($this->rawBlocks)) {
            $result = $this->restoreRawContent($result);
        }

        // If there are any footer lines that need to get added to a template we will
        // add them here at the end of the template. This gets used mainly for the
        // template inheritance via the extends keyword that should be appended.
        if (! empty($this->footer)) {
            $result = $this->addFooters($result);
        }

        return str_replace(
            ['##BEGIN-COMPONENT-CLASS##', '##END-COMPONENT-CLASS##'],
            '',
            $result
        );
    }

    /**
     * @param  string  $value
     * @return string
     */
    protected function storeUncompiledBlocks($value): string
    {
        if (Str::contains($value, '@verbatim')) {
            $value = $this->storeVerbatimBlocks($value);
        }

        if (Str::contains($value, '@php')) {
            $value = $this->storePhpBlocks($value);
        }

        return $value;
    }

    /**
     * Compile the component tags.
     *
     * @param  string  $value
     * @return string
     */
    protected function compileComponentTags(string $value): string
    {
        if (! $this->compilesComponentTags) {
            return $value;
        }

        return (new ComponentTagCompiler(
            $this->classComponentAliases,
            $this->classComponentNamespaces,
            $this
        ))->compile($value);
    }

    /**
     * Register a new precompiler.
     *
     * @param  callable  $precompiler
     * @return void
     */
    public function precompiler(callable $precompiler)
    {
        $this->precompilers[] = $precompiler;
    }

    /**
     * Register a class-based component alias directive.
     *
     * @param  string  $class
     * @param  string|null  $alias
     * @param  string  $prefix
     * @return void
     */
    public function component($class, $alias = null, $prefix = '')
    {
        if (! is_null($alias) && Str::contains($alias, '\\')) {
            [$class, $alias] = [$alias, $class];
        }

        if (is_null($alias)) {
            $alias = Str::contains($class, '\\View\\Components\\') ? collect(
                explode('\\', Str::after($class, '\\View\\Components\\'))
            )->map(
                function ($segment) {
                    return Str::kebab($segment);
                }
            )->implode(':') : Str::kebab(class_basename($class));
        }

        if (! empty($prefix)) {
            $alias = $prefix.'-'.$alias;
        }

        $this->classComponentAliases[$alias] = $class;
    }

    /**
     * Register an array of class-based components.
     *
     * @param  array  $components
     * @param  string  $prefix
     * @return void
     */
    public function components(array $components, $prefix = '')
    {
        foreach ($components as $key => $value) {
            if (is_numeric($key)) {
                static::component($value, null, $prefix);
            } else {
                static::component($key, $value, $prefix);
            }
        }
    }

    /**
     * Get the registered class component aliases.
     *
     * @return array
     */
    public function getClassComponentAliases()
    {
        return $this->classComponentAliases;
    }

    /**
     * Register a class-based component namespace.
     *
     * @param  string  $namespace
     * @param  string  $prefix
     * @return void
     */
    public function componentNamespace($namespace, $prefix)
    {
        $this->classComponentNamespaces[$prefix] = $namespace;
    }

    /**
     * Get the registered class component namespaces.
     *
     * @return array
     */
    public function getClassComponentNamespaces()
    {
        return $this->classComponentNamespaces;
    }

    /**
     * Register a component alias directive.
     *
     * @param  string  $path
     * @param  string|null  $alias
     * @return void
     */
    public function aliasComponent($path, $alias = null)
    {
        $alias = $alias ?: Arr::last(explode('.', $path));

        $this->directive(
            $alias,
            function ($expression) use ($path) {
                return $expression ? "<?php \$__env->startComponent('{$path}', {$expression}); ?>"
                    : "<?php \$__env->startComponent('{$path}'); ?>";
            }
        );

        $this->directive(
            'end'.$alias,
            function ($expression) {
                return '<?php echo $__env->renderComponent(); ?>';
            }
        );
    }

    public function compile($path = null)
    {
        if ($path) {
            $this->setPath($path);
        }

        if (! is_null($this->cachePath)) {
            $contents = $this->compileString($this->files->get($this->getPath()));

            if (! empty($this->getPath())) {
                $contents = $this->appendFilePath($contents);
            }

            $this->files->put(
                $this->getCompiledPath($this->getPath()),
                $contents
            );
        }
    }

    /**
     * @param  string|string[]|mixed  $contents
     * @return string
     */
    protected function appendFilePath($contents): string
    {
        $tokens = $this->getOpenAndClosingPhpTokens($contents);

        if ($tokens->isNotEmpty() && $tokens->last() !== T_CLOSE_TAG) {
            $contents .= ' ?>';
        }

        return $contents."<?php /**PATH {$this->getPath()} ENDPATH**/ ?>";
    }

    /**
     * Get the open and closing PHP tag tokens from the given string.
     *
     * @param  string  $contents
     * @return \Illuminate\Support\Collection
     */
    protected function getOpenAndClosingPhpTokens($contents)
    {
        return collect(token_get_all($contents))
            ->map(function ($tokens) {
                return $tokens[0];
            })
            ->filter(function ($token) {
                return in_array($token, [T_OPEN_TAG, T_OPEN_TAG_WITH_ECHO, T_CLOSE_TAG]);
            });
    }

    /**
     * compile statements
     *
     * @param  string  $value
     *
     * @return string|string[]|null
     */
    protected function compileStatements($value)
    {
        return preg_replace_callback(
            '/\B@(@?\w+(?:::\w+)?)([ \t]*)(\( ( (?>[^()]+) | (?3) )* \))?/x',
            function ($match) {
                return $this->compileStatement($match);
            },
            $value
        );
    }

    /**
     * compile custom extensions
     *
     * @param  string  $value
     *
     * @return string
     */
    protected function compileExtensions($value)
    {
        foreach ($this->extensions as $compiler) {
            $value = $compiler($value, $this);
        }

        return $value;
    }

    /**
     * Compile a statement
     *
     * @param  array  $match
     *
     * @return mixed|string
     */
    protected function compileStatement($match)
    {
        if (Str::contains($match[1], '@')) {
            $match[0] = isset($match[3]) ? $match[1].$match[3] : $match[1];
        } elseif (isset($this->customDirectives[$match[1]])) {
            $match[0] = $this->callCustomDirective($match[1], Arr::get($match, 3));
        } elseif (method_exists($this, $method = 'compile'.ucfirst($match[1]))) {
            $match[0] = $this->$method(Arr::get($match, 3));
        }

        return isset($match[3]) ? $match[0] : $match[0].$match[2];
    }

    /**
     * register an alias include
     *
     * @param  string  $path
     * @param  string|null  $alias
     */
    public function include($path, mixed $alias = null): void
    {
        $this->aliasInclude($path, $alias);
    }

    /**
     * register an alias include
     *
     * @param  string  $path
     * @param  string|null  $alias
     */
    public function aliasInclude(string $path, ?string $alias = null): void
    {
        $alias = $alias ?: Arr::last(explode('.', $path));

        $this->directive($alias, function ($expression) use ($path) {
            $expression = $this->stripParentheses($expression) ?: '[]';

            return "<?php echo \$__env->make('{$path}', {$expression}, \Illuminate\Support\Arr::except(get_defined_vars(), ['__data', '__path']))->render(); ?>";
        });
    }

    /**
     * Register an anonymous component namespace.
     */
    public function anonymousComponentNamespace(string $directory, string $prefix = null): void
    {
        $prefix ??= $directory;

        $this->anonymousComponentNamespaces[$prefix] = Str::of($directory)
            ->replace('/', '.')
            ->trim('. ')
            ->toString();
    }

    /**
     * @return array<string, string>
     */
    public function getAnonymousComponentNamespaces(): array
    {
        return $this->anonymousComponentNamespaces;
    }
}
