<?php

namespace Inside\Support\SerializableClosure;

class ClosureStream
{
    /**
     * The stream protocol.
     */
    public const STREAM_PROTO = 'inside-serializable-closure';

    /**
     * Checks if this stream is registered.
     */
    protected static bool $isRegistered = false;

    /**
     * The stream content.
     */
    protected string $content;

    /**
     * The stream content.
     */
    protected int $length;

    /**
     * The stream pointer.
     */
    protected int $pointer = 0;

    /**
     * Opens file or URL.
     */
    public function stream_open(string $path, string $mode, string $options, ?string &$opened_path): bool
    {
        $this->content = "<?php\nreturn ".substr($path, strlen(static::STREAM_PROTO.'://')).';';
        $this->length = strlen($this->content);

        return true;
    }

    /**
     * Read from stream.
     */
    public function stream_read(int $count): string
    {
        $value = substr($this->content, $this->pointer, $count);

        $this->pointer += $count;

        return $value;
    }

    /**
     * Tests for end-of-file on a file pointer.
     */
    public function stream_eof(): bool
    {
        return $this->pointer >= $this->length;
    }

    /**
     * Change stream options.
     */
    public function stream_set_option(int $option, int $arg1, int $arg2): bool
    {
        return false;
    }

    /**
     * Retrieve information about a file resource.
     */
    public function stream_stat(): bool|array
    {
        $stat = stat(__FILE__);
        // @phpstan-ignore-next-line
        $stat[7] = $stat['size'] = $this->length;

        return $stat;
    }

    /**
     * Retrieve information about a file.
     */
    public function url_stat(string $path, int $flags): bool|array
    {
        $stat = stat(__FILE__);
        // @phpstan-ignore-next-line
        $stat[7] = $stat['size'] = $this->length;

        return $stat;
    }

    /**
     * Seeks to specific location in a stream.
     */
    public function stream_seek(int $offset, int $whence = SEEK_SET): bool
    {
        $crt = $this->pointer;

        switch ($whence) {
            case SEEK_SET:
                $this->pointer = $offset;
                break;
            case SEEK_CUR:
                $this->pointer += $offset;
                break;
            case SEEK_END:
                $this->pointer = $this->length + $offset;
                break;
        }

        if ($this->pointer < 0 || $this->pointer >= $this->length) {
            $this->pointer = $crt;

            return false;
        }

        return true;
    }

    /**
     * Retrieve the current position of a stream.
     */
    public function stream_tell(): int
    {
        return $this->pointer;
    }

    /**
     * Registers the stream.
     */
    public static function register(): void
    {
        if (! static::$isRegistered) {
            static::$isRegistered = stream_wrapper_register(static::STREAM_PROTO, __CLASS__);
        }
    }
}
