<?php

namespace Inside\Support\Requirements;

use Illuminate\Support\Collection;

class SortedRequirements extends Collection
{
    /**
     * @param array $priorityMap
     * @param array|Collection $requirements
     */
    public function __construct(array $priorityMap, $requirements)
    {
        parent::__construct();

        if ($requirements instanceof Collection) {
            $requirements = $requirements->all();
        }

        $this->items = $this->sortRequirements($priorityMap, $requirements);
    }

    protected function sortRequirements(array $priorityMap, array $requirements): array
    {
        $lastIndex = 0;

        foreach ($requirements as $index => $requirement) {
            if (! is_string($requirement)) {
                continue;
            }

            $priorityIndex = $this->priorityMapIndex($priorityMap, $requirement);

            if (! is_null($priorityIndex)) {
                if (isset($lastPriorityIndex) && $priorityIndex < $lastPriorityIndex) {
                    return $this->sortRequirements(
                        $priorityMap,
                        array_values($this->moveRequirement($requirements, $index, $lastIndex))
                    );
                }
                $lastIndex = $index;

                $lastPriorityIndex = $priorityIndex;
            }
        }

        return $this->uniqueRequirements($requirements);
    }

    /**
     * @param array $priorityMap
     * @param string $middleware
     * @return int|string|null
     */
    protected function priorityMapIndex(array $priorityMap, string $middleware)
    {
        foreach ($this->requirementNames($middleware) as $name) {
            $priorityIndex = array_search($name, $priorityMap);

            if ($priorityIndex !== false) {
                return $priorityIndex;
            }
        }

        return null;
    }

    /**
     * @param string $requirement
     * @return \Generator
     */
    protected function requirementNames(string $requirement): \Generator
    {
        $stripped = head(explode(':', $requirement));

        yield $stripped;

        $interfaces = @class_implements($stripped);

        if ($interfaces !== false) {
            foreach ($interfaces as $interface) {
                yield $interface;
            }
        }
    }

    /**
     * @param array $requirements
     * @param int $from
     * @param int $to
     * @return array
     */
    protected function moveRequirement(array $requirements, int $from, int $to): array
    {
        array_splice($requirements, $to, 0, $requirements[$from]);

        unset($requirements[$from + 1]);

        return $requirements;
    }

    protected function uniqueRequirements(array $requirements): array
    {
        $seen = [];
        $result = [];

        foreach ($requirements as $requirement) {
            $key = \is_object($requirement) ? \spl_object_id($requirement) : $requirement;

            if (! isset($seen[$key])) {
                $seen[$key] = true;
                $result[] = $requirement;
            }
        }

        return $result;
    }
}
