<?php

namespace Inside\Support;

use Illuminate\Support\Arr;
use Symfony\Component\HttpFoundation\File\Exception\AccessDeniedException;
use Symfony\Component\HttpFoundation\File\Exception\FileNotFoundException;
use Symfony\Component\Mime\FileinfoMimeTypeGuesser;

/**
 * Class MimeTypeGuesser
 */
class MimeTypeGuesser extends FileinfoMimeTypeGuesser
{
    use HasMimeTypes;

    protected array $wrongDetectedMimes = [
        'text/xml' => [
            'file_binary_detectable' => false,
        ],
        'application/CDFV2' => [
            'file_binary_detectable' => true,
            'excepts'                => ['rvt'],
        ],
        'text/plain' => [
            'file_binary_detectable' => false,
        ],
        'application/zip' => [
            'file_binary_detectable' => false,
        ],
        'application/octet-stream' => [
            'file_binary_detectable' => false,
        ],
        'application/vnd.ms-office' => [
            'file_binary_detectable' => false,
        ],
    ];

    public function guessMimeType(string $path): ?string
    {
        if (! is_file($path)) {
            throw new FileNotFoundException($path);
        }

        if (! is_readable($path)) {
            throw new AccessDeniedException($path);
        }

        $mimeType = parent::guessMimeType($path);
        if ($mimeType && array_key_exists($mimeType, $this->wrongDetectedMimes)) {
            // Fallback to extension detection
            $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));

            // In case we got a wrong detected mimes
            if (! windows_os() && $this->wrongDetectedMimes[$mimeType]['file_binary_detectable']
                && is_array(
                    $this->wrongDetectedMimes[$mimeType]['excepts']
                )
                && ! in_array($extension, $this->wrongDetectedMimes[$mimeType]['excepts'])
            ) {
                return null; // fallback to BinaryMimeTypeGuesser
            }

            if (! empty($mimes = $this->getMimeTypes($extension))) {
                return Arr::first($mimes); // Get first mime type from our extension/mime map
            }

            return null; // We did not find the mime type
        }

        return $mimeType;
    }
}
