<?php

namespace Inside\Support;

use ErrorException;
use Exception;
use Illuminate\Contracts\Support\Arrayable;
use Illuminate\Contracts\Support\Jsonable;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\View;
use Illuminate\Support\Str;
use Inside\Application;
use Inside\Exceptions\JsonEncodingException;
use Inside\Providers\ServiceProvider;
use JsonSerializable;

class InsidePackage implements Arrayable, Jsonable, JsonSerializable
{
    protected ?Application $app = null;

    protected string $rawVersion;

    protected string $commit;

    protected string $major;

    protected string $minor;

    protected string $patch;

    protected bool $isDeferred = false;

    protected bool $isPrimary = false;

    protected array $providers = [];

    protected array $routes = [];

    protected array $channels = [];

    protected array $consoles = [];

    protected array $dependencies = [];

    protected array $path = [
        'inside' => 'vendor/',
        'drupal-module' => 'modules/custom/',
    ];

    protected bool $hasLanguage = false;

    protected bool $hasCustomViews = false;

    protected bool $hasDatabaseMigrations = false;

    protected bool $providersRegistered = false;

    protected bool $routesLoaded = false;

    protected bool $channelsLoaded = false;

    protected bool $consoleLoaded = false;

    protected bool $languageLoaded = false;

    protected bool $viewPathAdded = false;

    protected bool $migrationPathAdded = false;

    protected array $backofficeEntries = [];

    protected array $loadedBackofficeEntries = [];

    protected int $order = 0;

    public function __construct(
        protected string $name,
        protected string $version,
        protected array $namespaces = [],
        protected string $type = 'inside'
    ) {
        $this->rawVersion = $version;
        $this->isDeferred = ($name === 'maecia/'.config('app.code').'-back');
        $this->isPrimary = ($name === 'maecia/inside');
        [$this->version, $this->commit] = explode('@', $version);
        $semVerRegEx = '/(?<Major>0|(?:[1-9]\d*))(?:\.(?<Minor>0|(?:[1-9]\d*))(?:\.(?<Patch>0|(?:[1-9]\d*)))?(?:\-(?<PreRelease>[0-9A-Z\.-]+))?(?:\+(?<Meta>[0-9A-Z\.-]+))?)?/';
        $matches = [];
        if (! Str::startsWith($this->version, 'dev-') && preg_match($semVerRegEx, $this->version, $matches)) {
            [$this->major, $this->minor, $this->patch] = [$matches['Major'], $matches['Minor'], $matches['Patch']];
        }

        $this->hasCustomViews = File::exists(cms_base_path('vendor/'.$name.'/resources/views'));

        if ($this->isPrimary()) {
            $name = $name.'/core';
            $this->hasCustomViews = true;
        }

        $this->hasLanguage = File::exists(cms_base_path('vendor/'.$name.'/resources/lang'));
        $this->hasDatabaseMigrations = File::exists(cms_base_path('vendor/'.$name.'/database/migrations'));
    }

    public function getApp(): ?Application
    {
        return $this->app;
    }

    public function setApplication(?Application $app): self
    {
        $this->app = $app;

        return $this;
    }

    public function getRawVersion(): string
    {
        return $this->rawVersion;
    }

    public function setRawVersion(string $rawVersion): self
    {
        $this->rawVersion = $rawVersion;

        return $this;
    }

    public function getName(): string
    {
        if ($this->type != 'inside') {
            return str_replace('maecia/', '', $this->name);
        }

        return $this->name;
    }

    public function setName(string $name): self
    {
        $this->name = $name;

        return $this;
    }

    public function getVersion(): string
    {
        return $this->version;
    }

    public function setVersion(string $version): self
    {
        $this->version = $version;

        return $this;
    }

    public function getCommit(): string
    {
        return $this->commit;
    }

    public function setCommit(string $commit): self
    {
        $this->commit = $commit;

        return $this;
    }

    public function getMajor(): string
    {
        return $this->major;
    }

    public function setMajor(string $major): self
    {
        $this->major = $major;

        return $this;
    }

    public function getMinor(): string
    {
        return $this->minor;
    }

    public function setMinor(string $minor): self
    {
        $this->minor = $minor;

        return $this;
    }

    public function getPatch(): string
    {
        return $this->patch;
    }

    public function setPatch(string $patch): void
    {
        $this->patch = $patch;
    }

    public function isDeferred(): bool
    {
        return $this->isDeferred;
    }

    public function setIsDeferred(bool $isDeferred): self
    {
        $this->isDeferred = $isDeferred;

        return $this;
    }

    public function isPrimary(): bool
    {
        return $this->isPrimary;
    }

    public function setIsPrimary(bool $isPrimary): self
    {
        $this->isPrimary = $isPrimary;

        return $this;
    }

    public function getProviders(): array
    {
        return $this->providers;
    }

    public function setProviders(array $providers): self
    {
        $this->providers = $providers;

        return $this;
    }

    public function getBackofficeEntries(): array
    {
        return $this->backofficeEntries;
    }

    public function getLoadedBackofficeEntries(): array
    {
        $backofficeEntries = [];
        foreach ($this->loadedBackofficeEntries as $entry) {
            $backofficeEntries[] = $entry->getName().
                (in_array($entry->getName(), $this->backofficeEntries) ? ' (*) ' : ' (-) ').
                '['.implode(',', $entry->getRequirements()).']';
        }

        return $backofficeEntries;
    }

    public function setLoadedBackofficeEntries(array $entries): self
    {
        $this->loadedBackofficeEntries = $entries;

        return $this;
    }

    public function setBackofficeEntries(array $entries): self
    {
        $this->backofficeEntries = $entries;

        return $this;
    }

    public function getRoutes(): array
    {
        return $this->routes;
    }

    public function setRoutes(array $routes): self
    {
        $this->routes = $routes;

        return $this;
    }

    public function getChannels(): array
    {
        return $this->channels;
    }

    public function setChannels(array $channels): self
    {
        $this->channels = $channels;

        return $this;
    }

    public function getConsoles(): array
    {
        return $this->consoles;
    }

    public function setConsoles(array $consoles): self
    {
        $this->consoles = $consoles;

        return $this;
    }

    public function hasLanguage(): bool
    {
        return $this->hasLanguage;
    }

    public function setHasLanguage(bool $hasLanguage): self
    {
        $this->hasLanguage = $hasLanguage;

        return $this;
    }

    public function hasCustomViews(): bool
    {
        return $this->hasCustomViews;
    }

    public function setHasCustomViews(bool $hasCustomViews): self
    {
        $this->hasCustomViews = $hasCustomViews;

        return $this;
    }

    public function hasDatabaseMigrations(): bool
    {
        return $this->hasDatabaseMigrations;
    }

    public function setHasDatabaseMigrations(bool $hasDatabaseMigrations): self
    {
        $this->hasDatabaseMigrations = $hasDatabaseMigrations;

        return $this;
    }

    /**
     * @throws Exception
     */
    public function registerProviders(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->providersRegistered) {
            foreach ($this->providers as $provider) {
                try {
                    if (class_exists($provider)) {
                        $this->app->register($provider, true);
                    }
                } catch (Exception $exception) {
                    Log::error('[register]'.$provider.' => '.$exception->getMessage());
                }
            }
            $this->providersRegistered = true;
        }
    }

    /**
     * @throws Exception
     */
    public function bootProviders(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if ($this->providersRegistered) {
            foreach ($this->providers as $provider) {
                try {
                    if (! class_exists($provider)) {
                        continue;
                    }
                    if (is_string($provider)) {
                        $provider = new $provider($this->app);
                    }

                    if ($provider instanceof ServiceProvider) {
                        $provider->callBootingCallbacks();
                    }

                    if (method_exists($provider, 'boot')) {
                        $this->app->call([$provider, 'boot']);
                    }

                    if ($provider instanceof ServiceProvider) {
                        $provider->callBootedCallbacks();
                    }
                } catch (ErrorException $exception) {
                    if ($exception->getSeverity() != E_DEPRECATED) {
                        if (! is_string($provider)) {
                            $provider = get_class($provider);
                        }
                        Log::error('[boot] Error on ['.$provider.'] => '.$exception->getMessage());
                    }
                } catch (Exception $exception) {
                    if (! is_string($provider)) {
                        $provider = get_class($provider);
                    }
                    Log::error('[boot] Exception ['.$provider.'] => '.$exception->getMessage());
                }
            }
        }
    }

    /**
     * @throws Exception
     */
    public function loadRoutes(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->routesLoaded) {
            foreach ($this->routes as $route) {
                if (file_exists($route)) {
                    Route::group([], fn () => require $route);
                }
            }
            $this->routesLoaded = true;
        }
    }

    /**
     * load package channels for broadcasting
     * @throws Exception
     */
    public function loadChannels(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->channelsLoaded) {
            foreach ($this->channels as $channel) {
                if (file_exists($channel)) {
                    require $channel;
                }
            }
            $this->channelsLoaded = true;
        }
    }

    /**
     * Load package console commands for
     * @throws Exception
     */
    public function loadConsoles(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->app->runningInConsole()) {
            return;
        }
        if (! $this->consoleLoaded) {
            foreach ($this->consoles as $console) {
                if (file_exists($console)) {
                    require $console;
                }
            }
            $this->consoleLoaded = true;
        }
    }

    private function loadResourcesPath(string $folder): void
    {
        $isView = $folder === 'views';
        $path = null;
        if ($this->isPrimary()) {
            foreach (File::directories('vendor/maecia/inside') as $directory) {
                if (File::isDirectory($directory.'/resources/'.$folder)) {
                    $path = realpath(cms_base_path($directory.'/resources/'.$folder));
                    $isView ? View::addLocation($path) : Lang::addJsonPath($path);
                }
            }
        } else {
            $path = realpath(cms_base_path('vendor/'.$this->name.'/resources/'.$folder));
            $isView ? View::addLocation($path) : Lang::addJsonPath($path);
        }
    }

    /**
     * @throws Exception
     */
    public function loadLanguages(): void
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->languageLoaded) {
            if ($this->hasLanguage()) {
                $this->loadResourcesPath('lang');
            }
            $this->languageLoaded = true;
        }
    }

    /**
     * @throws Exception
     */
    public function addViewPath(): self
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->viewPathAdded) {
            if ($this->hasCustomViews()) {
                $this->loadResourcesPath('views');
            }
            $this->viewPathAdded = true;
        }

        return $this;
    }

    /**
     * @throws Exception
     */
    public function addMigrationPath(): self
    {
        if ($this->app === null) {
            throw new Exception('Application not set');
        }
        if (! $this->migrationPathAdded) {
            if ($this->hasDatabaseMigrations()) {
                $this->app->afterResolving('migrator', function ($migrator) {
                    $migrator->path(realpath(cms_base_path('vendor/'.$this->name.'/database/migrations')));
                });
            }
            $this->migrationPathAdded = true;
        }

        return $this;
    }

    public function getPath(): string
    {
        return $this->path[$this->type];
    }

    public function getNamespaces(): array
    {
        return $this->namespaces;
    }

    public function setNamespaces(array $namespaces): self
    {
        $this->namespaces = $namespaces;

        return $this;
    }

    public function getType(): string
    {
        return $this->type;
    }

    public function setType(string $type): self
    {
        $this->type = $type;

        return $this;
    }

    public function getDependencies(): array
    {
        return $this->dependencies;
    }

    public function setDependencies(array $dependencies = []): self
    {
        $this->dependencies = $dependencies;

        return $this;
    }

    public function setOrder(int $order): self
    {
        $this->order = $order;

        return $this;
    }

    public function getOrder(): int
    {
        return $this->order;
    }

    public function toArray(): array
    {
        return [
            'name' => $this->getName(),
            'rawVersion' => $this->getRawVersion(),
            'version' => $this->getVersion(),
            'commit' => $this->getCommit(),
            'isDeferred' => $this->isDeferred(),
            'isPrimary' => $this->isPrimary(),
            'hasLanguage' => $this->hasLanguage(),
            'hasCustomViews' => $this->hasCustomViews(),
            'hasMigrations' => $this->hasDatabaseMigrations(),
            'providers' => $this->getProviders(),
            'routes' => $this->getRoutes(),
            'channels' => $this->getChannels(),
            'consoles' => $this->getConsoles(),
            'backofficeEntries' => $this->getLoadedBackofficeEntries(),
        ];
    }

    public function jsonSerialize(): array
    {
        return $this->toArray();
    }

    /**
     * @throws JsonEncodingException
     */
    public function toJson($options = 0): string
    {
        $json = json_encode($this->jsonSerialize(), $options);

        if ($json === false || JSON_ERROR_NONE !== json_last_error()) {
            throw JsonEncodingException::forPackage($this, json_last_error_msg());
        }

        return $json;
    }

    /**
     * Convert this inside package to its string representation.
     *
     * @return string
     * @throws Exception
     */
    public function __toString()
    {
        return $this->toJson() ?: '';
    }
}
