<?php

namespace Inside\Services;

use Doctrine\DBAL\Schema\AbstractSchemaManager;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;

/**
 * Class Schema
 *
 * Class helper to get mysql schema info
 */
class SchemaService
{
    /**
     * @var AbstractSchemaManager
     */
    private $manager;

    public function __construct(AbstractSchemaManager $manager)
    {
        $this->manager = $manager;
    }

    /**
     * @return AbstractSchemaManager
     */
    public function getManager(): AbstractSchemaManager
    {
        return $this->manager;
    }

    /**
     * List table columns
     *
     * @param string $table
     * @return mixed
     */
    public function columns(string $table)
    {
        return Cache::remember(
            "inside_{$table}_columns",
            $this->lifetime(),
            function () use ($table) {
                $columns = $this->manager->listTableColumns($table);
                $keys = array_keys($columns);
                $values = array_values($columns);

                $keys = array_map(
                    function ($key) {
                        return trim(str_replace(['`', '"'], '', $key));
                    },
                    $keys
                );

                return array_combine($keys, $values);
            }
        );
    }

    /**
     * List table indexes
     *
     * @param string $table
     * @return mixed
     */
    public function indexes(string $table)
    {
        return Cache::remember(
            "inside_{$table}_indexes",
            $this->lifetime(),
            function () use ($table) {
                return $this->manager->listTableIndexes($table);
            }
        );
    }

    /**
     * list only indexed columns
     *
     * @param string $table
     *
     * @return array
     */
    public function indexedColumns(string $table): array
    {
        return Cache::remember(
            "inside_{$table}_indexed_columns",
            $this->lifetime(),
            function () use ($table) {
                $indexedColumns = array_reduce(
                    $this->indexes($table),
                    function ($indexedColumns, $index) {
                        return array_merge($indexedColumns, $index->getColumns());
                    },
                    []
                );

                $indexedColumns = array_unique($indexedColumns);
                sort($indexedColumns);

                return $indexedColumns;
            }
        );
    }

    /**
     * list table foreign keys
     *
     * @param string $table
     * @return mixed
     */
    public function foreignKeys(string $table)
    {
        return Cache::remember(
            "inside_{$table}_foreign_keys",
            $this->lifetime(),
            function () use ($table) {
                return $this->manager->listTableForeignKeys($table);
            }
        );
    }

    /**
     * Cache lifetime.
     *
     * @return Carbon
     */
    protected function lifetime(): Carbon
    {
        return now()->addSeconds(config('cache.schema.lifetime', 300));
    }
}
