<?php

namespace Inside\Services\Monitor\Checkers;

use Illuminate\Contracts\Queue\Factory as QueueFactory;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Queue;
use Predis\Collection\Iterator\Keyspace;
use Predis\Pipeline\Pipeline;
use Throwable;

class WorkerChecker extends Checker implements CheckerInterface
{
    protected array $workerQueueMap;

    public function isSupported(): bool
    {
        return Queue::getConnectionName() == 'redis';
    }

    public function prepareCheck(array $checkers): void
    {
        parent::prepareCheck($checkers);
        $this->workerQueueMap = [
            'default' => [
                get_high_priority_queue_name(),
                get_default_queue_name(),
                get_low_priority_queue_name(),
            ],
            'indexation' => [
                config('scout.queue.queue'),
            ],
        ];
    }

    public function processRunCheck(): bool
    {
        $connection = app(QueueFactory::class)->connection(Queue::getConnectionName());
        $redis = $connection->getRedis()->connection(config('queue.connections.redis.connection'));

        $this->lastCheckData['workers'] = [];
        foreach ($this->workerQueueMap as $group => $queues) {
            $this->lastCheckData['workers'][$group]['queues'] = [];
            foreach ($queues as $queue) {
                $this->lastCheckData['workers'][$group]['queues'][$queue] = [
                    'size' => Queue::size($queue),
                    'primary' => $redis->llen(Queue::getQueue($queue)),
                    'delayed' => $redis->zcount(Queue::getQueue($queue).':delayed', '-inf', '+inf'),
                    'reserved' => $redis->zcount(Queue::getQueue($queue).':reserved', '-inf', '+inf'),
                ];
            }
            try {
                $this->lastCheckData['workers'][$group]['failed'] =
                    DB::table(config('queue.failed.table'))->whereIn('queue', $queues)->count();
            } catch (Throwable $exception) {
            }
        }

        return true;
    }

    public function isHealthy(): bool
    {
        foreach ($this->lastCheckData['workers'] as $group => $checker) {
            if ($checker['checker'] === null) {
                return false;
            }
            $queue = array_key_first($checker['queues']);
            $maxExecutingTimeKey = 'inside.max-executing-time.'.$queue;
            $maxExecutingTime = 6 + Queue::size($queue) * 6;
            // worker may take long ( reindexing whole site should not take 12 hours ! )
            if ($maxExecutingTime > 720) {
                $maxExecutingTime = 720;
            }
            if (Queue::size($queue) === 0) {
                Cache::forget($maxExecutingTimeKey);
            } else {
                if (
                    Cache::has($maxExecutingTimeKey)
                    && ($savedMaxExecutingTime = Cache::get($maxExecutingTimeKey)) < $maxExecutingTime
                ) {
                    $maxExecutingTime = $savedMaxExecutingTime;
                } else {
                    Cache::put($maxExecutingTimeKey, $maxExecutingTime, 10);
                }
            }
            $lastWorker = get_date($checker['checker'], Carbon::RFC2822);
            if ($lastWorker && $lastWorker->diffInMinutes() > $maxExecutingTime) {
                // It takes too long !!
                return false;
            }
        }

        return true;
    }

    public function scan(mixed $client, string $pattern = '*', int $count = 100): mixed
    {
        $keys = [];

        foreach (new Keyspace($client->client(), $pattern) as $item) {
            $keys[] = $item;

            if (count($keys) == $count) {
                break;
            }
        }

        $script = <<<'LUA'
        local type = redis.call('type', KEYS[1])
        local ttl = redis.call('ttl', KEYS[1])
        return {KEYS[1], type, ttl}
LUA;

        return $client->pipeline(
            function (Pipeline $pipe) use ($keys, $script) {
                foreach ($keys as $key) {
                    $pipe->eval($script, 1, $key);
                }
            }
        );
    }

    public function fetch(mixed $client, string $key): ?array
    {
        if (! $client->exists($key)) {
            return [];
        }

        $type = $client->type($key)->__toString();
        $value = null;
        switch ($type) {
            case 'list':
                $value = $client->lrange($key, 0, -1);
        }

        return $value;
    }
}
