<?php

namespace Inside\Services\Monitor\Checkers;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Facades\Package;
use Inside\Support\Str;
use Linfo\Exceptions\FatalException;
use Linfo\Linfo;
use Throwable;

/**
 * Class EnvironmentChecker
 */
class EnvironmentChecker extends Checker implements CheckerInterface
{
    use HasCommand;

    /**
     * @return bool
     */
    public function processRunCheck(): bool
    {
        $this->lastCheckData['debug'] = config('app.debug');
        $this->lastCheckData['version'] = inside_version();
        $this->lastCheckData['code'] = config('app.code');
        $this->lastCheckData['url'] = config('app.url');
        $this->lastCheckData['server'] = windows_os() ? 'IIS' : 'Apache';

        try {
            $systemInfo = new Linfo(
                [
                    'temps' => [
                        'hwmon' => true,
                    ],
                    'show' => [
                        'kernel' => true,
                        'os' => true,
                        'ram' => true,
                        'mounts' => true,
                        'webservice' => true,
                        'phpversion' => true,
                        'uptime' => true,
                        'cpu' => true,
                        'distro' => true,
                        'model' => true,
                        'temps' => true,
                        'virtualization' => true,

                        'duplicate_mounts' => false,
                        'mounts_options' => false,
                    ],
                ]
            );
            $systemInfo = $systemInfo->getParser();

            $this->lastCheckData['system'] = [
                'os' => $systemInfo->getOS() ?? 'unknown',
                'kernel' => $systemInfo->getKernel() ?? 'unknown',
                'architecture' => $systemInfo->getCPUArchitecture() ?? 'unknown',
                'web' => $systemInfo->getWebService() ?? 'unknown',
                'php' => $systemInfo->getPhpVersion() ?? 'unknown',
                'distro' => array_merge($this->getSystemInfo(), $systemInfo->getDistro() ?: [
                    'name' => 'unknown',
                    'version'=>  'unknown',
                ]),
                'model' => $systemInfo->getModel() ?? 'unknown',
                'cpu' => $systemInfo->getCPU() ?? 'unknown',
                'memory' => $systemInfo->getRam() ?? 'unknown',
                'temps' => $systemInfo->getTemps() ?? 'unkown',
                //'disk'           => $systemInfo->getMounts() ?? 'unknown',
            ];
            $this->lastCheckData['system']['virtualization'] = $systemInfo->getVirtualization() ?? 'unknown';
        } catch (FatalException $exception) {
        }
        $this->lastCheckData['server_time'] = now()->toRfc2822String();
        try {
            $this->lastCheckData['active_tokens'] = DB::table('inside_authentication_tokens')->where(
                'last_used_at',
                '>=',
                now()->subMinutes(InsideSessionLifetime::getSessionLifetime())
            )->count();
        } catch (Throwable $exception) {
        }
        if (file_exists(storage_path('framework/inside_installed'))) {
            $lastInstalled = file_get_contents(storage_path('framework/inside_installed'));
            $lastInstalled = json_decode($lastInstalled, true);
            $this->lastCheckData['installed'] =
                get_date($lastInstalled['installed'])?->toRfc2822String();
            $this->lastCheckData['last_update'] =
                get_date($lastInstalled['updated'])?->toRfc2822String();
        }
        $this->lastCheckData['packages'] = Package::list();

        return true;
    }

    public function isHealthy(): bool
    {
        return isset($this->lastCheckData['debug']) && ! $this->lastCheckData['debug'];
    }

    protected function getSystemInfo(): array
    {
        $result = [];
        $this->setCommand('lsb_release -a');
        if ($this->commandRun()) {
            $matches = [];
            if (preg_match_all('/([^:]+):\t(.*)\n/', $this->getCommandResult(), $matches) > 0) {
                foreach ($matches[1] as $key => $keyString) {
                    $result[Str::snake(Str::lower($keyString))] = $matches[2][$key];
                }
            }
        }

        return $result;
    }
}
