<?php

namespace Inside\Services;

class AliasLoader
{
    /**
     * Indicates if a loader has been registered.x
     */
    protected bool $registered = false;

    /**
     * The namespace for all real-time facades.
     */
    protected static string $facadeNamespace = 'Facades\\';

    /**
     * The singleton instance of the loader.
     */
    protected static ?AliasLoader $instance = null;

    /**
     * Create a new AliasLoader instance.
     *
     * @param  array  $aliases
     * @return void
     */
    private function __construct(
        protected array $aliases = []
    ) {
    }

    /**
     * Get or create the singleton alias loader instance.
     */
    public static function getInstance(array $aliases = []): self
    {
        if (is_null(static::$instance)) {
            return static::$instance = new static($aliases);
        }

        $aliases = array_merge(static::$instance->getAliases(), $aliases);

        static::$instance->setAliases($aliases);

        return static::$instance;
    }

    /**
     * Load a class alias if it is registered.
     */
    public function load(string $alias): ?bool
    {
        if (static::$facadeNamespace && str_starts_with($alias, static::$facadeNamespace)) {
            $this->loadFacade($alias);

            return true;
        }

        if (isset($this->aliases[$alias])) {
            return class_alias($this->aliases[$alias], $alias);
        }

        return null;
    }

    /**
     * Load a real-time facade for the given alias.x
     */
    protected function loadFacade(string $alias): void
    {
        require $this->ensureFacadeExists($alias);
    }

    /**
     * Ensure that the given alias has an existing real-time facade class.
     */
    protected function ensureFacadeExists(string $alias): string
    {
        if (is_file($path = storage_path('framework/cache/facade-'.sha1($alias).'.php'))) {
            return $path;
        }

        /** @var string $fileContents */
        $fileContents = file_get_contents(__DIR__.'/stubs/facade.stub');

        file_put_contents($path, $this->formatFacadeStub(
            $alias,
            $fileContents
        ));

        return $path;
    }

    /**
     * Format the facade stub with the proper namespace and class.
     */
    protected function formatFacadeStub(string $alias, string $stub): string
    {
        $replacements = [
            str_replace('/', '\\', dirname(str_replace('\\', '/', $alias))),
            class_basename($alias),
            substr($alias, strlen(static::$facadeNamespace)),
        ];

        return str_replace(
            ['DummyNamespace', 'DummyClass', 'DummyTarget'],
            $replacements,
            $stub
        );
    }

    /**
     * Add an alias to the loader.
     */
    public function alias(string $alias, string $class): void
    {
        $this->aliases[$alias] = $class;
    }

    /**
     * Register the loader on the autoloader stack.
     */
    public function register(): void
    {
        if (! $this->registered) {
            $this->prependToLoaderStack();

            $this->registered = true;
        }
    }

    /**
     * Prepend the load method to the autoloader stack.x
     */
    protected function prependToLoaderStack(): void
    {
        spl_autoload_register([$this, 'load'], true, true);
    }

    /**
     * Get the registered aliases.
     */
    public function getAliases(): array
    {
        return $this->aliases;
    }

    /**
     * Set the registered aliases.
     */
    public function setAliases(array $aliases): void
    {
        $this->aliases = $aliases;
    }

    /**
     * Indicates if the loader has been registered.
     *
     * @return bool
     */
    public function isRegistered(): bool
    {
        return $this->registered;
    }

    /**
     * Set the "registered" state of the loader.
     */
    public function setRegistered(bool $value): void
    {
        $this->registered = $value;
    }

    /**
     * Set the real-time facade namespace.
     */
    public static function setFacadeNamespace(string $namespace): void
    {
        static::$facadeNamespace = rtrim($namespace, '\\').'\\';
    }

    /**
     * Set the value of the singleton alias loader.
     */
    public static function setInstance(self $loader): void
    {
        static::$instance = $loader;
    }

    /**
     * Clone method.
     *
     * @return void
     */
    private function __clone()
    {
    }
}
