<?php

namespace Inside\Providers;

use Illuminate\Routing\Router;
use Illuminate\Support\Facades\Route;
use Inside\Application;
use Inside\Support\RouteFormatter;
use Inside\Support\Traits\ForwardsCalls;
use Laravel\Lumen\Routing\UrlGenerator;

/**
 * Inside route service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RouteServiceProvider extends ServiceProvider
{
    use ForwardsCalls;

    /**
     * @var Application
     */
    protected $app;

    /**
     * @var string|null
     */
    protected ?string $namespace = null;

    public function register(): void
    {
        $this->booted(function () {
            if ($this->routesAreCached()) {
                $this->loadCachedRoutes();
            } else {
                $this->loadRoutes();
            }
        });
        $this->app->bind(
            RouteFormatter::class,
            function () {
                return new RouteFormatter();
            }
        );
    }

    public function boot(): void
    {
    }

    public function loadRoutes(): void
    {
        $this->mapApiRoutes();

        $this->mapWebRoutes();
    }

    protected function mapWebRoutes(): void
    {
        Route::group(
            ['namespace' => $this->namespace],
            function ($router) {
                require inside_core_path('routes/web.php');
            }
        );
    }

    protected function mapApiRoutes(): void
    {
        Route::group(
            ['namespace' => $this->namespace, 'prefix' => 'api/v1'],
            function ($router) {
                require inside_core_path('routes/api.php');
            }
        );
    }

    /**
     * routes are cached ?
     *
     * @return bool
     */
    protected function routesAreCached(): bool
    {
        return $this->app->routesAreCached();
    }

    /**
     * load cached routes
     */
    protected function loadCachedRoutes(): void
    {
        $this->app->booted(function () {
            require $this->app->getCachedRoutesPath();
        });
    }

    /**
     * @param string $method
     * @param array $parameters
     * @return mixed
     */
    public function __call(string $method, array $parameters)
    {
        return $this->forwardCallTo(
            $this->app->make(Router::class),
            $method,
            $parameters
        );
    }
}
