<?php

namespace Inside\Providers;

use Barryvdh\DomPDF\ServiceProvider as DomPDFServiceProvider;
use Closure;
use Doctrine\DBAL\Schema\AbstractSchemaManager;
use Illuminate\Contracts\Bus\Dispatcher as DispatcherContract;
use Illuminate\Contracts\Bus\QueueingDispatcher as QueueingDispatcherContract;
use Illuminate\Contracts\Queue\Factory as QueueFactoryContract;
use Illuminate\Contracts\Validation\ValidatesWhenResolved;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Database\Query\Builder as QueryBuilder;
use Illuminate\Database\Query\Expression;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;
use Inside\Application;
use Inside\Cache\RateLimiter;
use Inside\Console\CliDumper;
use Inside\Content\Providers\ChunkUploadServiceProvider;
use Inside\Database\FastPaginate\BuilderMixin;
use Inside\Database\FastPaginate\RelationMixin;
use Inside\Database\FastPaginate\ScoutMixin;
use Inside\Events\InsideBooted;
use Inside\Exceptions\MissingAppKeyException;
use Inside\Facades\Inside;
use Inside\Facades\Monitor;
use Inside\Horizon\Connectors\RedisConnector;
use Inside\Host\Exodus\Services\ContentTypeStatusService;
use Inside\Host\Exodus\Services\DrupalContainer;
use Inside\Http\HtmlDumper;
use Inside\Http\Requests\FormRequest;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Jobs\Bus\Dispatcher;
use Inside\Jobs\Bus\PendingDispatch;
use Inside\Jobs\Bus\Queueable;
use Inside\Listeners\AddJavascriptsToFrontend;
use Inside\Services\InsideApplicationService;
use Inside\Services\Monitor\Checkers\DiskChecker;
use Inside\Services\Monitor\Checkers\EnvironmentChecker;
use Inside\Services\Monitor\Checkers\MemoryChecker;
use Inside\Services\Monitor\Checkers\PartitionChecker;
use Inside\Services\Monitor\Checkers\PublicAccessChecker;
use Inside\Services\Monitor\Checkers\SystemChecker;
use Inside\Services\Monitor\Checkers\UserDirectoryChecker;
use Inside\Services\Monitor\Checkers\WorkerChecker;
use Inside\Services\Monitor\Monitoring;
use Inside\Services\PackageService;
use Inside\Services\SchemaService;
use Inside\Support\MimeTypeGuesser as InsideMimeTypeGuesser;
use Inside\Support\SerializableClosure\SerializableClosure;
use Inside\Support\ServerTiming;
use Inside\Support\Str;
use Inside\Support\Traits\Localizable;
use Inside\Support\Version;
use Maatwebsite\Excel\ExcelServiceProvider;
use Mews\Purifier\PurifierServiceProvider;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use ReflectionException;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Mime\FileBinaryMimeTypeGuesser;
use Symfony\Component\Mime\MimeTypes;

/**
 * Inside package discover service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AppServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     *
     * @return void
     */
    public function boot(): void
    {
        // View https://stackoverflow.com/questions/23786359/laravel-migration-unique-key-is-too-long-even-if-specified
        Schema::defaultStringLength(191);

        // Load package service so PackageServiceProvider can work
        $this->app->singleton(
            'inside.package',
            function ($app) {
                return new PackageService($app);
            }
        );

        // Our default lang
        Lang::addJsonPath(realpath(__DIR__.'/../../resources/custom/lang'));
        // Note: Other modules lang will be add during package boot/registration

        // Load automatcs
        Lang::addJsonPath(realpath(cms_base_path('vendor/maecia/inside/lang/automatics')));

        $this->loadLanguageOverrides();

        // FormRequest system
        $this->app->afterResolving(ValidatesWhenResolved::class, function ($resolved) {
            $resolved->validateResolved();
        });

        $this->app->resolving(FormRequest::class, function ($request, $app) {
            /** @var FormRequest $request */
            $request = FormRequest::createFrom($app['request'], $request);

            $request->setContainer($app);
        });

        // Add useful validators
        /**
         * @deprecated
         */
        Validator::extend(
            'uuid',
            function ($attribute, $value, $parameters, $validator) {
                if (! is_string($value)) {
                    return false;
                }

                return preg_match('/^[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}$/iD', $value) > 0;
            }
        );

        // Remove when update to Laravel 9
        Collection::macro('reduceWithKeys', function ($callback, $initial = null) {
            $result = $initial;

            // @phpstan-ignore-next-line
            foreach ($this as $key => $value) {
                $result = $callback($initial, $value, $key);
            }

            return $result;
        });

        if (config('database.default') === 'pgsql') {
            DB::statement("SELECT setval('file_managed_fid_seq', (SELECT MAX(fid) FROM file_managed));");
        }

        Builder::macro('orderByField', function (string $field, array $values) {
            /** @var Builder $this */
            return $this->orderByRaw(str('CASE ? END')->replace('?', implode(' ', array_map(function ($value, $order) use ($field) {
                return str('WHEN ?=? THEN ?')->replaceArray('?', [$field, Str::wrap($value, '\''), $order]);
            }, $values, range(1, count($values))))));
        });

        QueryBuilder::macro('orderByField', function (string $field, array $values) {
            /** @var Builder $this */
            return $this->orderByRaw(str('CASE ? END')->replace('?', implode(' ', array_map(function ($value, $order) use ($field) {
                return str('WHEN ?=? THEN ?')->replaceArray('?', [$field, Str::wrap($value, '\''), $order]);
            }, $values, range(1, count($values))))));
        });

        Builder::macro('orderByDate', function (string $field, string $format, string $direction) {
            /** @var Builder $this */
            return match (config('database.default')) {
                'pgsql' => $this->orderByRaw(str('TO_CHAR(?, ?) ?')->replaceArray('?', [$field, Str::wrap($format, '\''), $direction])),
                default => $this->orderByRaw(str('DATE_FORMAT(?, ?) ?')->replaceArray('?', [$field, Str::wrap($format, '\''), $direction])),
            };
        });

        QueryBuilder::macro('orderByDate', function (string $field, string $format, string $direction) {
            /** @var Builder $this */
            return match (config('database.default')) {
                'pgsql' => $this->orderByRaw(str('TO_CHAR(?, ?) ?')->replaceArray('?', [$field, Str::wrap($format, '\''), $direction])),
                default => $this->orderByRaw(str('DATE_FORMAT(?, ?) ?')->replaceArray('?', [$field, Str::wrap($format, '\''), $direction])),
            };
        });

        Builder::macro('whereLike', function (Closure|string|array|Expression $column, mixed $value, bool $negate = false) {
            /** @var Builder $this */
            return match (config('database.default')) {
                'pgsql' => $this->where($column, $negate ? 'NOT ILIKE' : 'ILIKE', $value),
                default => $this->where($column, $negate ? 'NOT LIKE' : 'LIKE', $value),
            };
        });

        QueryBuilder::macro('whereLike', function (Closure|string|array|Expression $column, mixed $value, bool $negate = false) {
            /** @var QueryBuilder $this */
            return match (config('database.default')) {
                'pgsql' => $this->where($column, $negate ? 'NOT ILIKE' : 'ILIKE', $value),
                default => $this->where($column, $negate ? 'NOT LIKE' : 'LIKE', $value),
            };
        });

        Builder::macro('whereRawLike', function (string $raw, string $search, bool $negate = false) {
            /** @var Builder $this */
            return match (config('database.default')) {
                'pgsql' => $this->whereRaw($raw.($negate ? ' NOT ILIKE ?' : ' ILIKE ?'), $search),
                default => $this->whereRaw($raw.($negate ? ' NOT LIKE ?' : ' LIKE ?'), $search),
            };
        });

        QueryBuilder::macro('whereRawLike', function (string $raw, string $search, bool $negate = false) {
            /** @var QueryBuilder $this */
            return match (config('database.default')) {
                'pgsql' => $this->whereRaw($raw.($negate ? ' NOT ILIKE ?' : ' ILIKE ?'), $search),
                default => $this->whereRaw($raw.($negate ? ' NOT LIKE ?' : ' LIKE ?'), $search),
            };
        });

        Builder::macro('orWhereLike', function (Closure|string|array|Expression $column, mixed $value, bool $negate = false) {
            /** @var Builder $this */
            return match (config('database.default')) {
                'pgsql' => $this->orWhere($column, $negate ? 'NOT ILIKE' : 'ILIKE', $value),
                default => $this->orWhere($column, $negate ? 'NOT LIKE' : 'LIKE', $value),
            };
        });

        QueryBuilder::macro('orWhereLike', function (Closure|string|array|Expression $column, mixed $value, bool $negate = false) {
            /** @var QueryBuilder $this */
            return match (config('database.default')) {
                'pgsql' => $this->orWhere($column, $negate ? 'NOT ILIKE' : 'ILIKE', $value),
                default => $this->orWhere($column, $negate ? 'NOT LIKE' : 'LIKE', $value),
            };
        });

        // Register Validation Macro
        Request::macro('validate', function (array $rules, ...$params) {
            // @phpstan-ignore-next-line
            return validator($this->all(), $rules, ...$params)->validate();
        });

        Request::macro('validateWithBag', function (string $errorBag, array $rules, ...$params) {
            try {
                // @phpstan-ignore-next-line
                return $this->validate($rules, ...$params);
            } catch (ValidationException $e) {
                $e->errorBag = $errorBag;

                throw $e;
            }
        });

        // Add cache rate limiter
        $this->app->singleton(RateLimiter::class, function ($app) {
            return new RateLimiter($app->make('cache')->driver($app['config']->get('cache.limiter')));
        });

        // Use our own Redis Queue Connector
        Queue::addConnector('redis', function () {
            return new RedisConnector($this->app['redis']);
        });

        // Add front to layout
        Event::listen(InsideBooted::class, AddJavascriptsToFrontend::class);

        if (config('session.secure')) {
            URL::forceScheme('https');
        }

        // Add our Mimeguesser
        // Reset Mime guesser, drupal mess everything
        MimeTypes::setDefault(new MimeTypes());
        MimeTypes::getDefault()->registerGuesser(new FileBinaryMimeTypeGuesser());
        MimeTypes::getDefault()->registerGuesser(new InsideMimeTypeGuesser());

        // register monitor checkers
        Monitor::register(DiskChecker::class);
        Monitor::register(PartitionChecker::class);
        Monitor::register(MemoryChecker::class);
        Monitor::register(SystemChecker::class);
        Monitor::register(EnvironmentChecker::class);
        Monitor::register(WorkerChecker::class);
        Monitor::register(PublicAccessChecker::class);
        Monitor::register(UserDirectoryChecker::class);

        $this->app->singleton(DrupalContainer::class, fn () =>  new DrupalContainer());
        $this->app->bind(ContentTypeStatusService::class, fn () => new ContentTypeStatusService());

        $this->app->register(MailServiceProvider::class);

        // Register view service provider
        $this->app->register(ViewServiceProvider::class);

        $this->registerBus();

        $this->registerDumper();
    }

    /**
     * Register the application services.
     *
     * @return void
     * @throws ReflectionException
     */
    public function register(): void
    {
        try {
            $this->mergeConfigTo(__DIR__.'/../../config/scheduler.php', 'scheduler');
            $this->mergeConfigTo(__DIR__.'/../../config/session.php', 'session');
            $this->mergeConfigTo(__DIR__.'/../../config/security.php', 'security');
            $this->mergeConfigFrom(__DIR__.'/../../config/workers.php', 'workers');
            $this->mergeConfigTo(__DIR__.'/../../config/debugbar.php', 'debugbar');
            $this->mergeRecursiveConfigFrom(__DIR__.'/../../config/authentication.php', 'authentication');
        } catch (NotFoundExceptionInterface|ContainerExceptionInterface) {
        }

        $this->app->bind(
            'inside',
            function ($app) {
                return new InsideApplicationService();
            }
        );
        class_alias(Inside::class, 'Inside');
        class_alias(Dispatchable::class, \Illuminate\Foundation\Bus\Dispatchable::class);
        class_alias(Queueable::class, \Illuminate\Bus\Queuable::class);
        class_alias(PendingDispatch::class, \Illuminate\Foundation\Bus\PendingDispatch::class);
        class_alias(Localizable::class, \Illuminate\Support\Traits\Localizable::class);

        $this->app->alias('mailer', \Illuminate\Contracts\Mail\Mailer::class);

        // Register special inside-mysql driver
        /**
         * $this->app->bind(
         * 'db.connector.inside-mysql', function ($app) {
         * return new MysqlConnection;
         * }
         * );
         */
        $this->app->singleton(Version::class, function ($app) {
            return new Version($app);
        });

        // Register external services
        $this->app->register(ExcelServiceProvider::class);
        $this->app->register(DomPDFServiceProvider::class);
        $this->app->register(PurifierServiceProvider::class);
        $this->app->register(ChunkUploadServiceProvider::class);

        $this->app->singleton(
            SchemaService::class,
            function (Application $app) {
                /** @var AbstractSchemaManager|null $schema */
                $schema = $app['db']->connection()->getDoctrineSchemaManager();

                if ($schema) {
                    $platform = $schema->getDatabasePlatform();
                    $platform->registerDoctrineTypeMapping('enum', 'string');
                    $platform->registerDoctrineTypeMapping('set', 'string');

                    return new SchemaService($schema);
                }

                return null;
            }
        );

        Builder::mixin(new BuilderMixin());
        Relation::mixin(new RelationMixin());

        if (class_exists(\Laravel\Scout\Builder::class)) {
            \Laravel\Scout\Builder::mixin(new ScoutMixin());
        }

        // Register monitoring service
        $this->app->bind(
            Monitoring::class,
            function ($app) {
                return new Monitoring();
            }
        );

        /**
         * Debug useful stuff ( only loaded if present & not in production
         */
        $this->loadDebugAndPerformanceTools();

        $this->registerSerializableClosureSecurityKey();
    }

    protected function loadLanguageOverrides(): void
    {
        // Load lang key languages from our project and override defaults ( useful for validation strings for example )
        if (File::exists(back_path('resources/lang')) && File::isDirectory(back_path('resources/lang'))) {
            foreach (list_languages() as $language) {
                if (File::exists(back_path('resources/lang/').$language)
                    && File::isDirectory(
                        back_path('resources/lang/').$language
                    )
                ) {
                    // Force load default languages
                    foreach (
                        Finder::create()->files()->name('*.php')->in(
                            [back_path('resources/lang/').$language, app('path.lang')]
                        ) as $file
                    ) {
                        $keys = Arr::dot(require $file, basename($file->getRealPath(), '.php').'.');
                        if (! empty($keys)) {
                            Lang::addLines($keys, $language, basename($file->getRealPath(), '.php'));
                        }
                    }
                }
            }
        }
    }

    protected function loadDebugAndPerformanceTools(): void
    {
        if ($this->app->environment() === 'local'
            && class_exists(\Barryvdh\LaravelIdeHelper\IdeHelperServiceProvider::class)
        ) {
            $this->mergeConfigFrom(__DIR__.'/../../config/ide-helper.php', 'ide-helper');
            $this->app->register(\Barryvdh\LaravelIdeHelper\IdeHelperServiceProvider::class);
        }
        if (env('USE_DEBUGBAR', false) && $this->app->environment() !== 'production'
            && class_exists(
                \Barryvdh\Debugbar\LumenServiceProvider::class
            )
        ) {
            $this->app->register(\Barryvdh\Debugbar\LumenServiceProvider::class);
        }
        $this->app->singleton(ServerTiming::class, function ($app) {
            return new ServerTiming(new \Symfony\Component\Stopwatch\Stopwatch());
        });
    }

    protected function registerBus(): void
    {
        $this->app->singleton(Dispatcher::class, function ($app) {
            return new Dispatcher($app, function ($connection = null) use ($app) {
                return $app[QueueFactoryContract::class]->connection($connection);
            });
        });

        $this->app->alias(
            Dispatcher::class,
            DispatcherContract::class
        );

        $this->app->alias(
            Dispatcher::class,
            QueueingDispatcherContract::class
        );
    }

    protected function registerSerializableClosureSecurityKey(): void
    {
        $config = $this->app->make('config')->get('app');

        if (! class_exists(SerializableClosure::class) || empty($config['key'])) {
            return;
        }

        SerializableClosure::setSecretKey($this->parseKey($config));
    }

    protected function parseKey(array $config): ?string
    {
        if (Str::startsWith($key = $this->key($config), $prefix = 'base64:')) {
            $key = base64_decode(Str::after($key, $prefix));
        }

        return $key;
    }

    protected function key(array $config): string
    {
        return tap($config['key'], function ($key) {
            if (empty($key)) {
                throw new MissingAppKeyException();
            }
        });
    }

    protected function registerDumper(): void
    {
        $basePath = $this->app->basePath();

        $compiledViewPath = $this->app['config']->get('view.compiled');

        $format = $_SERVER['VAR_DUMPER_FORMAT'] ?? null;

        match (true) {
            'html' == $format => HtmlDumper::register($basePath, $compiledViewPath),
            'cli' == $format => CliDumper::register($basePath, $compiledViewPath),
            'server' == $format => null,
            $format && 'tcp' == parse_url($format, PHP_URL_SCHEME) => null,
            default => in_array(PHP_SAPI, ['cli', 'phpdbg'])
                ? CliDumper::register($basePath, $compiledViewPath)
                : HtmlDumper::register($basePath, $compiledViewPath),
        };
    }
}
