<?php

namespace Inside\Logging;

use Closure;
use Illuminate\Support\Str;
use Inside\Application;
use InvalidArgumentException;
use Monolog\Formatter\FormatterInterface;
use Monolog\Formatter\LineFormatter;
use Monolog\Handler\ErrorLogHandler;
use Monolog\Handler\HandlerInterface;
use Monolog\Handler\MissingExtensionException;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\SlackWebhookHandler;
use Monolog\Handler\StreamHandler;
use Monolog\Handler\SyslogHandler;
use Monolog\Handler\WhatFailureGroupHandler;
use Monolog\Logger as Monolog;
use Psr\Log\LoggerInterface;
use Throwable;

class LogManager implements LoggerInterface
{
    use ParsesLogConfiguration;

    /**
     * The array of resolved channels.
     */
    protected array $channels = [];

    /**
     * The registered custom driver creators.
     */
    protected array $customCreators = [];

    /**
     * The standard date format to use when writing logs.
     */
    protected string $dateFormat = 'Y-m-d H:i:s';

    /**
     * Create a new Log manager instance.
     */
    public function __construct(
        protected Application $app
    ) {
    }

    /**
     * Build an on-demand log channel.
     */
    public function build(array $config): LoggerInterface
    {
        unset($this->channels['ondemand']);

        return $this->get('ondemand', $config);
    }

    /**
     * Create a new, on-demand aggregate logger instance.
     *
     * @param array $channels
     * @param string|null $channel
     * @return LoggerInterface
     */
    public function stack(array $channels, ?string $channel = null): LoggerInterface
    {
        return new Logger(
            $this->createStackDriver(compact('channels', 'channel')),
            $this->app['events']
        );
    }

    /**
     * Get a log channel instance.
     *
     * @param string|null $channel
     * @return Logger
     */
    public function channel(?string $channel = null): Logger
    {
        return $this->driver($channel);
    }

    /**
     * Get a log driver instance.
     *
     * @param string|null $driver
     * @return Logger
     */
    public function driver(?string $driver = null): Logger
    {
        return $this->get($driver ?? $this->getDefaultDriver());
    }

    /**
     * Attempt to get the log from the local cache.
     *
     * @param string $name
     * @return Logger
     */
    protected function get(string $name): Logger
    {
        try {
            return $this->channels[$name] ?? with(
                $this->resolve($name),
                function ($logger) use ($name) {
                    return $this->channels[$name] = $this->tap($name, new Logger($logger, $this->app['events']));
                }
            );
        } catch (Throwable $e) {
            return tap(
                $this->createEmergencyLogger(),
                function ($logger) use ($e) {
                    $logger->emergency(
                        'Unable to create configured logger. Using emergency logger.',
                        [
                            'exception' => $e,
                        ]
                    );
                }
            );
        }
    }

    /**
     * Apply the configured taps for the logger.
     *
     * @param string $name
     * @param Logger $logger
     * @return LoggerInterface
     */
    protected function tap(string $name, LoggerInterface $logger): LoggerInterface
    {
        foreach ($this->configurationFor($name)['tap'] ?? [] as $tap) {
            [$class, $arguments] = $this->parseTap($tap);

            $this->app->make($class)->__invoke($logger, ...explode(',', $arguments));
        }

        return $logger;
    }

    /**
     * Parse the given tap class string into a class name and arguments string.
     */
    protected function parseTap(string $tap): array
    {
        return Str::contains($tap, ':') ? explode(':', $tap, 2) : [$tap, ''];
    }

    /**
     * Create an emergency log handler to avoid white screens of death.
     */
    protected function createEmergencyLogger(): LoggerInterface
    {
        return new Logger(
            new Monolog(
                'laravel',
                $this->prepareHandlers(
                    [
                        new StreamHandler(
                            $this->app->storagePath().'/logs/laravel.log',
                            // @phpstan-ignore-next-line
                            $this->level(['level' => 'debug'])
                        ),
                    ]
                )
            ),
            $this->app['events']
        );
    }

    /**
     * Resolve the given log instance by name.
     *
     * @throws InvalidArgumentException
     */
    protected function resolve(string $name): LoggerInterface
    {
        // @phpstan-ignore-next-line
        $config ??= $this->configurationFor($name);

        if (is_null($config)) {
            throw new InvalidArgumentException("Log [{$name}] is not defined.");
        }

        if (isset($this->customCreators[$config['driver']])) {
            return $this->callCustomCreator($config);
        }

        $driverMethod = 'create'.ucfirst($config['driver']).'Driver';

        if (method_exists($this, $driverMethod)) {
            return $this->{$driverMethod}($config);
        }

        throw new InvalidArgumentException("Driver [{$config['driver']}] is not supported.");
    }

    /**
     * Call a custom driver creator.
     */
    protected function callCustomCreator(array $config): mixed
    {
        return $this->customCreators[$config['driver']]($this->app, $config);
    }

    /**
     * Create a custom log driver instance.
     */
    protected function createCustomDriver(array $config): LoggerInterface
    {
        $factory = is_callable($via = $config['via']) ? $via : $this->app->make($via);

        return $factory($config);
    }

    /**
     * Create an aggregate log driver instance.
     *
     * @param array $config
     * @return LoggerInterface
     */
    protected function createStackDriver(array $config): LoggerInterface
    {
        if (is_string($config['channels'])) {
            $config['channels'] = explode(',', $config['channels']);
        }

        $handlers = collect($config['channels'])->flatMap(
            function ($channel) {
                return $this->channel($channel)->getHandlers();
            }
        )->all();

        if ($config['ignore_exceptions'] ?? false) {
            $handlers = [new WhatFailureGroupHandler($handlers)];
        }

        return new Monolog($this->parseChannel($config), $handlers);
    }

    /**
     * Create an instance of the single file log driver.
     *
     * @throw InvalidArgumentException
     */
    protected function createSingleDriver(array $config): LoggerInterface
    {
        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    new StreamHandler(
                        $config['path'],
                        /** @phpstan-ignore-next-line  */
                        $this->level($config),
                        $config['bubble'] ?? true,
                        $config['permission'] ?? null,
                        $config['locking'] ?? false
                    ),
                    $config
                ),
            ]
        );
    }

    /**
     * Create an instance of the daily file log driver.
     */
    protected function createDailyDriver(array $config): LoggerInterface
    {
        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    new RotatingFileHandler(
                        $config['path'],
                        $config['days'] ?? 7,
                        /** @phpstan-ignore-next-line  */
                        $this->level($config),
                        $config['bubble'] ?? true,
                        $config['permission'] ?? null,
                        $config['locking'] ?? false
                    ),
                    $config
                ),
            ]
        );
    }

    /**
     * Create an instance of the Slack log driver.
     * @throws MissingExtensionException
     */
    protected function createSlackDriver(array $config): LoggerInterface
    {
        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    new SlackWebhookHandler(
                        $config['url'],
                        $config['channel'] ?? null,
                        $config['username'] ?? 'Laravel',
                        $config['attachment'] ?? true,
                        $config['emoji'] ?? ':boom:',
                        $config['short'] ?? false,
                        $config['context'] ?? true,
                        /** @phpstan-ignore-next-line  */
                        $this->level($config),
                        $config['bubble'] ?? true,
                        $config['exclude_fields'] ?? []
                    ),
                    $config
                ),
            ]
        );
    }

    /**
     * Create an instance of the syslog log driver.
     */
    protected function createSyslogDriver(array $config): LoggerInterface
    {
        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    new SyslogHandler(
                        Str::snake($this->app['config']['app.name'], '-'),
                        $config['facility'] ?? LOG_USER,
                        /** @phpstan-ignore-next-line  */
                        $this->level($config)
                    ),
                    $config
                ),
            ]
        );
    }

    /**
     * Create an instance of the "error log" log driver.
     */
    protected function createErrorlogDriver(array $config): LoggerInterface
    {
        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    new ErrorLogHandler(
                        $config['type'] ?? ErrorLogHandler::OPERATING_SYSTEM,
                        /** @phpstan-ignore-next-line  */
                        $this->level($config)
                    )
                ),
            ]
        );
    }

    /**
     * Create an instance of any handler available in Monolog.
     * @throws InvalidArgumentException
     */
    protected function createMonologDriver(array $config): Monolog|LoggerInterface
    {
        if (! is_a($config['handler'], HandlerInterface::class, true)) {
            throw new InvalidArgumentException(
                $config['handler'].' must be an instance of '.HandlerInterface::class
            );
        }

        $with = array_merge(
            ['level' => $this->level($config)],
            $config['with'] ?? [],
            $config['handler_with'] ?? []
        );

        return new Monolog(
            $this->parseChannel($config),
            [
                $this->prepareHandler(
                    /** @phpstan-ignore-next-line  */
                    $this->app->make($config['handler'], $with),
                    $config
                ),
            ]
        );
    }

    /**
     * Prepare the handlers for usage by Monolog.
     */
    protected function prepareHandlers(array $handlers): array
    {
        foreach ($handlers as $key => $handler) {
            $handlers[$key] = $this->prepareHandler($handler);
        }

        return $handlers;
    }

    /**
     * Prepare the handler for usage by Monolog.
     *
     * @param  HandlerInterface  $handler
     * @param array $config
     * @return HandlerInterface
     */
    protected function prepareHandler(HandlerInterface $handler, array $config = []): HandlerInterface
    {
        if (method_exists($handler, 'setFormatter')) {
            if (! isset($config['formatter'])) {
                $handler->setFormatter($this->formatter());
            } elseif ($config['formatter'] !== 'default') {
                $handler->setFormatter($this->app->make($config['formatter'], $config['formatter_with'] ?? []));
            }
        }

        return $handler;
    }

    /**
     * Get a Monolog formatter instance.
     */
    protected function formatter(): FormatterInterface
    {
        return tap(
            new LineFormatter(null, $this->dateFormat, true, true),
            function ($formatter) {
                $formatter->includeStacktraces();
            }
        );
    }

    /**
     * Get fallback log channel name.
     */
    protected function getFallbackChannelName(): string
    {
        return $this->app->bound('env') ? $this->app->environment() : 'production'; // @phpstan-ignore-line
    }

    /**
     * Get the log connection configuration.
     *
     * @param string $name
     * @return array|null
     */
    protected function configurationFor(string $name): ?array
    {
        return $this->app['config']["logging.channels.{$name}"];
    }

    /**
     * Get the default log driver name.
     *
     * @return string
     */
    public function getDefaultDriver(): string
    {
        return $this->app['config']['logging.default'];
    }

    /**
     * Set the default log driver name.
     *
     * @param string $name
     * @return void
     */
    public function setDefaultDriver(string $name): void
    {
        $this->app['config']['logging.default'] = $name;
    }

    /**
     * Register a custom driver creator Closure.
     */
    public function extend(string $driver, Closure $callback): static
    {
        $this->customCreators[$driver] = $callback->bindTo($this, $this);

        return $this;
    }

    /**
     * System is unusable.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function emergency($message, array $context = []): void
    {
        $this->driver()->emergency($message, $context);
    }

    /**
     * Action must be taken immediately.
     *
     * Example: Entire website down, database unavailable, etc. This should
     * trigger the SMS alerts and wake you up.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function alert($message, array $context = []): void
    {
        $this->driver()->alert($message, $context);
    }

    /**
     * Critical conditions.
     *
     * Example : Application component unavailable, unexpected exception.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function critical($message, array $context = []): void
    {
        $this->driver()->critical($message, $context);
    }

    /**
     * Runtime errors that do not require immediate action but should typically
     * be logged and monitored.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function error($message, array $context = []): void
    {
        $this->driver()->error($message, $context);
    }

    /**
     * Exceptional occurrences that are not errors.
     *
     * Example: Use of deprecated APIs, poor use of an API, undesirable things
     * that are not necessarily wrong.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function warning($message, array $context = []): void
    {
        $this->driver()->warning($message, $context);
    }

    /**
     * Normal but significant events.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function notice($message, array $context = []): void
    {
        $this->driver()->notice($message, $context);
    }

    /**
     * Interesting events.
     *
     * Example: User logs in, SQL logs.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function info($message, array $context = []): void
    {
        $this->driver()->info($message, $context);
    }

    /**
     * Detailed debug information.
     *
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function debug($message, array $context = []): void
    {
        $this->driver()->debug($message, $context);
    }

    /**
     * Logs with an arbitrary level.
     *
     * @param mixed $level
     * @param string $message
     * @param array $context
     *
     * @return void
     */
    public function log($level, $message, array $context = []): void
    {
        $this->driver()->log($level, $message, $context);
    }

    /**
     * Dynamically call the default driver instance.
     *
     * @param  string  $method
     * @param  array  $parameters
     * @return mixed
     */
    public function __call(string $method, array $parameters): mixed
    {
        return $this->driver()->$method(...$parameters);
    }
}
