<?php

namespace Inside\Jobs\Queue;

use Closure;
use DateTimeInterface;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Encryption\Encrypter;
use Illuminate\Queue\InvalidPayloadException;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\HigherOrderTapProxy;
use Inside\Events\JobQueued;
use Inside\Support\Contracts\Queue\ShouldBeEncrypted;
use Inside\Support\Str;
use const JSON_UNESCAPED_UNICODE;
use Throwable;

#[\AllowDynamicProperties]
class RedisQueue extends \Illuminate\Queue\RedisQueue
{
    /**
     * @throws Throwable
     */
    protected function createObjectPayload($job, $queue): array
    {
        $payload = $this->withCreatePayloadHooks($queue, [
            'uuid' => (string) Str::uuid(),
            'displayName' => $this->getDisplayName($job),
            'job' => 'Inside\Jobs\Queue\CallQueuedHandler@call',
            'maxTries' => $job->tries ?? null,
            'maxExceptions' => $job->maxExceptions ?? null,
            'failOnTimeout' => $job->failOnTimeout ?? false,
            'timeout' => $job->timeout ?? null,
            'retryUntil' => $this->getJobExpiration($job),
            'data' => [
                'commandName' => $job,
                'command' => $job,
            ],
        ]);

        try {
            $command = $this->jobShouldBeEncrypted($job) && $this->container->bound(Encrypter::class)
                ? $this->container[Encrypter::class]->encrypt(serialize(clone $job))
                : serialize(clone $job);
        } catch (Throwable $exception) {
            Log::error(__('[RedisQueue::createObjectPayload] :message (:name)', ['message' => $exception->getMessage(), 'name' => $this->getDisplayName($job)]));
            throw $exception;
        }

        return array_merge($payload, [
            'data' => array_merge($payload['data'], [
                'commandName' => get_class($job),
                'command' => $command,
            ]),
        ]);
    }

    protected function createStringPayload($job, $queue, $data): array
    {
        return $this->withCreatePayloadHooks($queue, [
            'uuid' => (string) Str::uuid(),
            'displayName' => is_string($job) ? explode('@', $job)[0] : null, //@phpstan-ignore-line
            'job' => $job,
            'maxTries' => null,
            'maxExceptions' => null,
            'failOnTimeout' => false,
            'backoff' => null,
            'timeout' => null,
            'data' => $data,
        ]);
    }

    protected function jobShouldBeEncrypted(object $job): bool
    {
        if ($job instanceof ShouldBeEncrypted) {
            return true;
        }

        return isset($job->shouldBeEncrypted) && $job->shouldBeEncrypted;
    }

    /**
     * @throws BindingResolutionException
     */
    public function push($job, $data = '', $queue = null): mixed
    {
        return $this->enqueueUsing(
            $job,
            $this->createPayload($job, $this->getQueue($queue), $data),
            $queue,
            null,
            function ($payload, $queue) {
                return $this->pushRaw($payload, $queue);
            }
        );
    }

    /**
     * @throws InvalidPayloadException
     */
    protected function createPayload($job, $queue, $data = ''): string
    {
        if ($job instanceof Closure) {
            $job = CallQueuedClosure::create($job);
        }

        // @phpstan-ignore-next-line
        $payload = json_encode($this->createPayloadArray($job, $queue, $data), JSON_UNESCAPED_UNICODE);

        if (json_last_error() !== JSON_ERROR_NONE || $payload === false) {
            throw new InvalidPayloadException(
                'Unable to JSON encode payload. Error code: '.json_last_error()
            );
        }

        return $payload;
    }

    /**
     * @param  mixed  $job
     * @param  string  $payload
     * @param  string|null  $queue
     * @param  mixed  $delay
     * @param  callable  $callback
     * @return HigherOrderTapProxy|mixed
     * @throws BindingResolutionException
     */
    protected function enqueueUsing($job, string $payload, ?string $queue, mixed $delay, callable $callback): mixed
    {
        if ($this->shouldDispatchAfterCommit($job) &&
            $this->container->bound('db.transactions')) {
            return $this->container->make('db.transactions')->addCallback(
                function () use ($payload, $queue, $delay, $callback, $job) {
                    return tap($callback($payload, $queue, $delay), function ($jobId) use ($job) {
                        $this->raiseJobQueuedEvent($jobId, $job);
                    });
                }
            );
        }

        return tap($callback($payload, $queue, $delay), function ($jobId) use ($job) {
            $this->raiseJobQueuedEvent($jobId, $job);
        });
    }

    /**
     * @param mixed $job
     * @return bool
     */
    protected function shouldDispatchAfterCommit(mixed $job): bool
    {
        if (is_object($job) && isset($job->afterCommit)) {
            return $job->afterCommit;
        }

        if (isset($this->dispatchAfterCommit)) {
            return $this->dispatchAfterCommit;
        }

        return false;
    }

    /**
     * @param  int|string|null  $jobId
     * @param mixed $job
     * @return void
     */
    protected function raiseJobQueuedEvent(int|string|null $jobId, mixed $job): void
    {
        if ($this->container->bound('events')) {
            $this->container['events']->dispatch(new JobQueued($this->connectionName, $jobId, $job));
        }
    }
}
