<?php

namespace Inside\Jobs\Queue;

use Exception;
use Illuminate\Contracts\Bus\Dispatcher;
use Illuminate\Contracts\Cache\Repository as Cache;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Encryption\Encrypter;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Pipeline\Pipeline;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\Jobs\Job;
use Illuminate\Support\Str;
use Inside\Contracts\Queue\ShouldBeUnique;
use Inside\Contracts\Queue\ShouldBeUniqueUntilProcessing;
use Inside\Jobs\Bus\Batchable;
use ReflectionClass;
use RuntimeException;
use Throwable;

class CallQueuedHandler
{
    public function __construct(
        protected Dispatcher $dispatcher,
        protected Container $container
    ) {
    }

    /**
     * Handle the queued job.
     * @throws BindingResolutionException
     */
    public function call(Job $job, array $data): void
    {
        try {
            $command = $this->setJobInstanceIfNecessary($job, $this->getCommand($data));
        } catch (ModelNotFoundException $e) {
            $this->handleModelNotFound($job, $e);

            return;
        }

        if ($command instanceof ShouldBeUniqueUntilProcessing) {
            $this->ensureUniqueJobLockIsReleased($command);
        }

        $this->dispatchThroughMiddleware($job, $command);

        if (! $job->isReleased() && ! $command instanceof ShouldBeUniqueUntilProcessing) {
            $this->ensureUniqueJobLockIsReleased($command);
        }

        if (! $job->hasFailed() && ! $job->isReleased()) {
            $this->ensureNextJobInChainIsDispatched($command);
            $this->ensureSuccessfulBatchJobIsRecorded($command);
        }

        if (! $job->isDeletedOrReleased()) {
            $job->delete();
        }
    }

    /**
     * Get the command from the given payload.
     *
     * @throws RuntimeException
     */
    protected function getCommand(array $data): mixed
    {
        if (Str::startsWith($data['command'], 'O:')) {
            return unserialize($data['command']);
        }

        if ($this->container->bound(Encrypter::class)) {
            return unserialize($this->container[Encrypter::class]->decrypt($data['command']));
        }

        throw new RuntimeException('Unable to extract job payload.');
    }

    /**
     * Dispatch the given job / command through its specified middleware.
     */
    protected function dispatchThroughMiddleware(Job $job, mixed $command): mixed
    {
        return (new Pipeline($this->container))->send($command)
            ->through(
                array_merge(
                    method_exists($command, 'middleware') ? $command->middleware() : [],
                    $command->middleware ?? []
                )
            )
            ->then(function ($command) use ($job) {
                return $this->dispatcher->dispatchNow(
                    $command,
                    $this->resolveHandler($job, $command)
                );
            });
    }

    /**
     * Resolve the handler for the given command.
     */
    protected function resolveHandler(Job $job, mixed $command): mixed
    {
        $handler = null;

        if ($this->dispatcher instanceof \Illuminate\Bus\Dispatcher) {
            $handler = $this->dispatcher->getCommandHandler($command) ?: null;
        }

        if ($handler) {
            $this->setJobInstanceIfNecessary($job, $handler);
        }

        return $handler;
    }

    /**
     * Set the job instance of the given class if necessary.
     */
    protected function setJobInstanceIfNecessary(Job $job, mixed $instance): mixed
    {
        if (in_array(InteractsWithQueue::class, class_uses_recursive($instance))) {
            $instance->setJob($job);
        }

        return $instance;
    }

    /**
     * Ensure the next job in the chain is dispatched if applicable.
     */
    protected function ensureNextJobInChainIsDispatched(mixed $command): void
    {
        if (method_exists($command, 'dispatchNextJobInChain')) {
            $command->dispatchNextJobInChain();
        }
    }

    /**
     * Ensure the batch is notified of the successful job completion.
     */
    protected function ensureSuccessfulBatchJobIsRecorded(mixed $command): void
    {
        $uses = class_uses_recursive($command);

        if (
            ! in_array(Batchable::class, $uses) ||
            ! in_array(InteractsWithQueue::class, $uses) ||
            is_null($command->batch())
        ) {
            return;
        }

        $command->batch()->recordSuccessfulJob($command->job->uuid());
    }

    /**
     * Ensure the lock for a unique job is released.
     * @throws BindingResolutionException
     */
    protected function ensureUniqueJobLockIsReleased(mixed $command): void
    {
        if (! $command instanceof ShouldBeUnique) {
            return;
        }

        $uniqueId = method_exists($command, 'uniqueId')
            ? $command->uniqueId()
            : '';

        $cache = method_exists($command, 'uniqueVia')
            ? $command->uniqueVia()
            : $this->container->make(Cache::class);

        $cache->lock(
            'laravel_unique_job:'.get_class($command).$uniqueId
        )->forceRelease();
    }

    /**
     * Handle a model not found exception.
     */
    protected function handleModelNotFound(Job $job, Exception $exception): void
    {
        $class = $job->resolveName();

        try {
            /** @phpstan-ignore-next-line */
            $shouldDelete = (new ReflectionClass($class))
                ->getDefaultProperties()['deleteWhenMissingModels'] ?? false;
        } catch (Exception $exception) {
            $shouldDelete = false;
        }

        if ($shouldDelete) {
            $job->delete();

            return;
        }

        $job->fail($exception);
    }

    /**
     * Call the failed method on the job instance.
     *
     * The exception that caused the failure will be passed.
     */
    public function failed(array $data, ?Exception $exception): void
    {
        $command = $this->getCommand($data);

        if (! $command instanceof ShouldBeUniqueUntilProcessing) {
            $this->ensureUniqueJobLockIsReleased($command);
        }

        $this->ensureFailedBatchJobIsRecorded('', $command, $exception);
        $this->ensureChainCatchCallbacksAreInvoked('', $command, $exception);

        if (method_exists($command, 'failed')) {
            $command->failed($exception);
        }
    }

    /**
     * Ensure the batch is notified of the failed job.
     */
    protected function ensureFailedBatchJobIsRecorded(string $uuid, mixed $command, ?Exception $exception): void
    {
        if (
            ! in_array(Batchable::class, class_uses_recursive($command)) ||
            is_null($command->batch())
        ) {
            return;
        }

        $command->batch()->recordFailedJob($uuid, $exception);
    }

    /**
     * Ensure the chained job catch callbacks are invoked.
     */
    protected function ensureChainCatchCallbacksAreInvoked(string $uuid, mixed $command, ?Exception $exception): void
    {
        if (method_exists($command, 'invokeChainCatchCallbacks')) {
            $command->invokeChainCatchCallbacks($exception);
        }
    }
}
