<?php

namespace Inside\Jobs\Queue;

use Closure;
use Illuminate\Contracts\Container\Container;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Inside\Jobs\Bus\Batchable;
use Inside\Jobs\Bus\Dispatchable;
use Inside\Jobs\Bus\Queueable;
use ReflectionException;
use ReflectionFunction;
use Throwable;

class CallQueuedClosure implements ShouldQueue
{
    use Batchable;
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * The callbacks that should be executed on failure.
     */
    public array $failureCallbacks = [];

    /**
     * Indicate if the job should be deleted when models are missing.
     */
    public bool $deleteWhenMissingModels = true;

    /**
     * Create a new job instance.
     */
    public function __construct(
        protected SerializableClosure $closure
    ) {
    }

    public static function create(Closure $job): self
    {
        return new self(new SerializableClosure($job));
    }

    /**
     * Execute the job.
     */
    public function handle(Container $container): void
    {
        $container->call($this->closure->getClosure(), ['job' => $this]);
    }

    public function onFailure(callable|Closure $callback): self
    {
        $this->failureCallbacks[] = $callback instanceof Closure
            ? new SerializableClosure($callback)
            : $callback;

        return $this;
    }

    /**
     * Handle a job failure.
     */
    public function failed(Throwable $exception): void
    {
        foreach ($this->failureCallbacks as $callback) {
            $callback($exception);
        }
    }

    /**
     * Get the display name for the queued job.
     * @throws ReflectionException
     */
    public function displayName(): string
    {
        $reflection = new ReflectionFunction($this->closure->getClosure());

        return 'Closure ('.basename($reflection->getFileName()).':'.$reflection->getStartLine().')';
    }
}
