<?php

namespace Inside\Http\Middleware\Headers;

use Closure;
use Illuminate\Http\Request;
use Inside\Facades\Package;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Content Security Policy Header middleware.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ContentSecurityPolicyMiddleware
{
    public function handle(Request $request, Closure $next): mixed
    {
        $reportURI = env('REPORT_URI', null);

        // TODO: temporary for dev on intranet-inside.com must be done for production etc...
        $proxyURI = 'proxy.'.preg_replace('#https?://#', '', config('app.url'));
        $isSecured = preg_match('#^https://#', config('app.url'));

        $policies = [
            'default-src'     => ['*'],
            'script-src'      => [
                '\'self\'',
                '\'unsafe-inline\'',
                '\'unsafe-eval\'',
                'data:',
                'https:',
                'http://*.hotjar.com',
                'https://*.newrelic.com',
                'https://teams.microsoft.com',
                'https://*.teams.microsoft.com',
                'https://www.googletagmanager.com',
                'https://*.google-analytics.com',
            ],
            'style-src'       => ['\'self\'', '\'unsafe-inline\'', 'https:'],
            'img-src'         => [
                '\'self\'',
                'blob:',
                'data:',
                'https:',
                'http://*.basemaps.cartocdn.com',
                'http://*.hotjar.com',
                'https://teams.microsoft.com',
                'https://*.teams.microsoft.com',
            ],
            'connect-src'     => [
                '\'self\'',
                'http://*.hotjar.com:*',
                'https://*.hotjar.com:*',
                'wss://*.hotjar.com',
                'https://*.hotjar.io:*',
                'https://httpbin.org',
                'https://bam.nr-data.net',
                'https://onesignal.com',
                'https://cdn.onesignal.com',
                'https://cdn.cookielaw.org',
                'https://*.google-analytics.com',
                'https://*.analytics.google.com',
                'https://matomo.intranet-inside.com',
                'https://maps.googleapis.com',
                'https://fonts.googleapis.com',
                'https://fonts.gstatic.com',
                ($isSecured ? 'https://' : 'http://').$proxyURI,
                ($isSecured ? 'wss://' : 'ws://').$proxyURI,
            ],
            'font-src'        => ['\'self\'', 'data:', 'https:'],
            'object-src'      => ['\'none\''],
            'media-src'       => ['\'self\'', 'https:'],
            'form-action'     => ['\'self\'', 'https://*.os.tc:*'],
            'frame-ancestors' => ['\'self\'', 'https://teams.microsoft.com',  'https://*.teams.microsoft.com'],
            'frame-src'       => ['\'self\'', 'https:'],
            'worker-src'      => ['\'self\'', 'https:'],
        ];
        $policies['worker-src'][] = 'blob:';

        //
        $policies['connect-src'][] = config('app.url').':6001';
        $policies['connect-src'][] = str_replace(['http', 'https'], ['ws', 'ws'], config('app.url')).':6001';

        if ($reportURI) {
            $policies[] = ['report-uri' => $reportURI];
        }

        foreach (config('security.policies', []) as $group => $customPolicies) {
            if (! is_array($policies[$group])) {
                $policies[$group] = [];
            }
            $policies[$group] = array_merge($policies[$group], $customPolicies);
        }

        $policies = collect($policies)->map(
            function (array $values, string $directive) {
                $valueString = implode(' ', $values);

                return empty($valueString) ? "{$directive}" : "{$directive} {$valueString}";
            }
        )->implode(';');

        $headers = [
            'Content-Security-Policy' => preg_replace('!\s+!', ' ', $policies),
            'Permissions-Policy'      => 'fullscreen=(self "https://indd.adobe.com")',
        ];

        $response = $next($request);

        if ($response instanceof BinaryFileResponse) {
            return $response;
        }
        foreach ($headers as $key => $value) {
            $response->headers->set($key, $value);
        }

        return $response;
    }
}
