<?php

declare(strict_types=1);

namespace Inside\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Facades\Package;
use Inside\Jobs\TestMail;
use Laravel\Lumen\Routing\Controller;
use OpenApi\Annotations as OA;

/**
 * Class AdminController
 * @OA\Tag(name="Admin",description="Les services liés au système disponible pour les admin maecia.")
 */
class AdminController extends Controller
{
    /**
     * Simply clear cache
     *
     *
     * @OA\Post(
     *    path="/admin/cache/clear",
     *     operationId="cacheClear",
     *     tags={"Admin"},
     *     summary="Supppression du cache",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function cacheClear(): JsonResponse
    {
        return response()->json($this->callArtisan('cache:clear'));
    }

    /**
     * Clear route cache
     */
    public function routeClear(): JsonResponse
    {
        return response()->json($this->callArtisan('inside:route:clear'));
    }

    /**
     * Get package list info
     *
     *
     * @OA\Get(
     *    path="/admin/package/list",
     *     operationId="packageList",
     *     tags={"Admin"},
     *     summary="Liste des modules inside installés avec leurs informations.",
     *     @OA\Response(
     *         response=200
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function getPackages(): JsonResponse
    {
        return response()->json(['data' => Package::list()]);
    }

    /**
     * Reload version
     *
     * @OA\Post(
     *    path="/admin/reload/version",
     *     operationId="reloadVersion",
     *     tags={"Admin"},
     *     summary="Rechargement de la version d'inside",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function reloadVersion(): JsonResponse
    {
        return response()->json($this->callArtisan('version:reload'));
    }

    /**
     * Rebuild search index
     *
     * @OA\Post(
     *    path="/admin/index/rebuild",
     *     operationId="indexRebuild",
     *     tags={"Admin"},
     *     summary="Demande de réindexation des contenus.",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function indexRebuild(): JsonResponse
    {
        return response()->json($this->callArtisan('index:rebuild'));
    }

    /**
     * Reindex missing contents
     *
     * @OA\Post(
     *    path="/admin/index/rebuild/missing",
     *     operationId="indexRebuildMissing",
     *     tags={"Admin"},
     *     summary="Reindex missing contents.",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function indexMissingContents(): JsonResponse
    {
        return response()->json($this->callArtisan('index:rebuild:missing'));
    }

    /**
     * Rebuild models dynamic classes
     *
     *
     * @OA\Post(
     *    path="/admin/models/rebuild",
     *     operationId="modelsRebuild",
     *     tags={"Admin"},
     *     summary="Regénération des classes de contenus métiers",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function modelsRebuild(): JsonResponse
    {
        return response()->json($this->callArtisan('models:rebuild'));
    }

    /**
     * Regenerate all images
     *
     *
     * @OA\Post(
     *    path="/admin/images/generate",
     *     operationId="imagesGenerate",
     *     tags={"Admin"},
     *     summary="Génération des images ( tous les styles )",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function imagesGenerate(): JsonResponse
    {
        Storage::disk('local')->deleteDirectory('styles');

        return response()->json($this->callArtisan('images:generate'));
    }

    /**
     * Clean ghosted contents
     *
     *
     * @OA\Post(
     *    path="/admin/inside/cleanup",
     *     operationId="cleanup",
     *     tags={"Admin"},
     *     summary="Demande de nettoyage d'inside ( contenu fantomes, pivots orphelins, utilisateurs désynchronisés)",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function cleanup(): JsonResponse
    {
        return response()->json($this->callArtisan('inside:cleanup'));
    }

    /**
     * Clean database
     *
     *
     * @OA\Post(
     *    path="/admin/inside/database/cleanup",
     *     operationId="databaseCleanup",
     *     tags={"Admin"},
     *     summary="Demande de nettoyage de la base de données.",
     *     @OA\Response(
     *         response=200,
     *         @OA\JsonContent(ref="#/components/schemas/AdminResponse")
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function cleanupDatabase(): JsonResponse
    {
        return response()->json();
    }

    /**
     * Sending an email to check mailer is properly working
     *
     * @OA\Post(
     *    path="/admin/email/test",
     *     operationId="emailTest",
     *     tags={"Admin"},
     *     summary="Envoi un email de test",
     *     @OA\Response(
     *         response=200
     *    )
     * )
     *
     * @return JsonResponse
     */
    public function testEmail(): JsonResponse
    {
        /** @var User $user */
        $user = Auth::user();
        try {
            Queue::push(new TestMail($user));
        } catch (Exception $e) {
            return response()->json(['errors' => $e->getMessage()]);
        }

        return response()->json(['status' => 'sent']);
    }

    /**
     * Call an artisan command
     *
     * @OA\Schema(
     *     schema="AdminResponse",
     *     title="Réponse d'une commande admin",
     *     type="object",
     *     description="La réponse d'une commande interne.",
     *     @OA\Property(
     *         property="success",
     *         title="Le succès",
     *         type="bool",
     *         description="Le succès de la commande effectuée",
     *         default=false,
     *         example=true
     *     ),
     *     @OA\Property(
     *         property="data",
     *         title="Les données",
     *         type="string",
     *         description="Le retour de la commande interne",
     *         default=null,
     *         example="Commande réussie"
     *     ),
     * )
     *
     * @param string $command
     * @return array
     */
    protected function callArtisan(string $command): array
    {
        return ['success' => Artisan::call($command) === 0, 'data' => Artisan::output()];
    }

    // Some futur ideas: enable/disable debug, enable/disable maintenance mode
}
