<?php

namespace Inside\Http\Controllers\Admin;

use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\File;
use Symfony\Component\HttpFoundation\BinaryFileResponse;

/**
 * Class LogController
 */
class LogController
{
    /**
     * export current log
     *
     * @return BinaryFileResponse
     */
    public function getDebugLogs(): BinaryFileResponse
    {
        $path = cms_base_path(
            env('APP_LOG_PATH', 'storage/logs').'/'.Carbon::now()->format('Y-m-d').'-inside.log'
        );
        if (! File::exists($path)) {
            throw new ModelNotFoundException();
        }

        return new BinaryFileResponse(
            $path,
            200,
            [
                'Content-Type' => 'text/plain',
            ]
        );
    }

    /**
     * get Logs to be displayed in /settings
     *
     * @return array|LengthAwarePaginator
     */
    public function getLogs(Request $request)
    {
        $path = cms_base_path(
            env('APP_LOG_PATH', 'storage/logs').'/'.Carbon::now()->format('Y-m-d').'-inside.log'
        );
        if (! File::exists($path)) {
            return [];
        }
        $log = [];
        $content = File::get($path);

        $headings = null;

        $logPattern = '/\[\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}([\+-]\d{4})?\].*/';

        preg_match_all($logPattern, $content, $headings);

        if (! is_array($headings)) {
            return [];
        }

        /** @var string[] $logData */
        $logData = preg_split($logPattern, $content);

        if ($logData && $logData[0] < 1) {
            array_shift($logData);
        }

        $levels = [
            'debug',
            'info',
            'notice',
            'warning',
            'error',
            'critical',
            'alert',
            'emergency',
            'processed',
            'failed',
        ];

        $filterLevel = $request->get('level');
        if (isset($filterLevel) && is_string($filterLevel)) {
            if (empty($filterLevel)) {
                unset($filterLevel);
            } else {
                $filterLevel = explode(',', $filterLevel);
            }
        }
        foreach ($headings as $h) {
            for ($i = 0, $j = count($h); $i < $j; $i++) {
                foreach ($levels as $level) {
                    if (strpos(strtolower($h[$i]), '.'.$level) || strpos(strtolower($h[$i]), $level.':')) {
                        preg_match(
                            "/^\[(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}([\+-]\d{4})?)\](?:.*?(\w+)\.|.*?)$level: (.*?)( in .*?:[0-9]+)?$/i",
                            $h[$i],
                            $current
                        );
                        if (! isset($current[4])) {
                            continue;
                        }
                        if (isset($filterLevel) && ! in_array($level, $filterLevel)) {
                            continue;
                        }

                        $inFile = null;
                        if (isset($current[5])) {
                            $inFile = [
                                'file' => '',
                                'line' => '',
                            ];
                            [$inFile['file'], $inFile['line']] =
                                explode(':', trim(str_replace('in ', '', $current[5])), 2);
                        }
                        $date = get_date($current[1], 'Y-m-d H:i:s');

                        $log[] = [
                            'context' => $current[3],
                            'level' => $level,
                            'date' => $date ? $date->timestamp : null,
                            'text' => $current[4],
                            'in_file' => $inFile,
                            'stack' => array_values(
                                array_filter(
                                    explode("\n", preg_replace("/^\n*/", "\n", $logData[$i]) ?: ''),
                                    function ($line) {
                                        return ! empty($line) && $line != 'Stack trace:';
                                    }
                                )
                            ),
                        ];
                    }
                }
            }
        }

        $logs = collect(array_reverse($log));

        $perPage = $request->get('perPage', 20);
        $page = LengthAwarePaginator::resolveCurrentPage('page');

        return new LengthAwarePaginator(
            $logs->forPage($page, $perPage),
            $logs->count(),
            $perPage,
            $page,
            [
                'path' => LengthAwarePaginator::resolveCurrentPath(),
                'pageName' => 'page',
            ]
        );
    }
}
