<?php

namespace Inside\Database\Schema;

use Illuminate\Database\Connection;
use Illuminate\Filesystem\Filesystem;
use Inside\Support\Process;

abstract class SchemaState
{
    /**
     * The connection instance.
     */
    protected Connection $connection;

    /**
     * The filesystem instance.
     */
    protected Filesystem $files;

    /**
     * The process factory callback.
     *
     * @var callable
     */
    protected $processFactory;

    /**
     * The output callable instance.
     *
     * @var callable
     */
    protected $output;

    /**
     * Create a new dumper instance.
     *
     * @param Connection      $connection
     * @param Filesystem|null $files
     * @param callable|null   $processFactory
     * @return void
     */
    public function __construct(Connection $connection, Filesystem $files = null, callable $processFactory = null)
    {
        $this->connection = $connection;

        $this->files = $files ?: new Filesystem();

        $this->processFactory = $processFactory
            ?: function (...$arguments) {
                return Process::fromShellCommandline(...$arguments);
            };

        $this->handleOutputUsing(
            function () {
                //
            }
        );
    }

    /**
     * Dump the database's schema into a file.
     *
     * @param string $path
     * @return void
     */
    abstract public function dump(string $path);

    /**
     * Load the given schema file into the database.
     *
     * @param string $path
     * @return void
     */
    abstract public function load(string $path);

    /**
     * Create a new process instance.
     *
     * @param array $arguments
     * @return Process
     */
    public function makeProcess(...$arguments): Process
    {
        return call_user_func($this->processFactory, ...$arguments);
    }

    /**
     * Specify the callback that should be used to handle process output.
     *
     * @param callable $output
     * @return $this
     */
    public function handleOutputUsing(callable $output)
    {
        $this->output = $output;

        return $this;
    }
}
