<?php

namespace Inside\Database\FastPaginate;

use Closure;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Query\Expression;
use Inside\Database\Eloquent\Builder;

class FastPaginate
{
    public function fastPaginate(): Closure
    {
        return $this->paginate('paginate', function (array $items, $paginator) {
            // @phpstan-ignore-next-line
            return $this->paginator(
                $items,
                $paginator->total(),
                $paginator->perPage(),
                $paginator->currentPage(),
                $paginator->getOptions()
            );
        });
    }

    public function simpleFastPaginate(): Closure
    {
        return $this->paginate('simplePaginate', function (array $items, $paginator) {
            // @phpstan-ignore-next-line
            return $this->simplePaginator(
                $items,
                $paginator->perPage(),
                $paginator->currentPage(),
                $paginator->getOptions()
            )->hasMorePagesWhen($paginator->hasMorePages());
        });
    }

    protected function paginate(string $paginationMethod, Closure $paginatorOutput): Closure
    {
        return function ($perPage = null, $columns = ['*'], $pageName = 'page', $page = null) use (
            $paginationMethod,
            $paginatorOutput
        ) {
            /** @var Builder $this */
            $base = $this->getQuery();
            if (filled($base->havings) || filled($base->groups)) {
                return $this->{$paginationMethod}($perPage, $columns, $pageName, $page);
            }

            /** @var Model $model */
            $model = $this->newModelInstance();
            $key = $model->getKeyName();
            $table = $model->getTable();

            try {
                $innerSelectColumns = self::getInnerSelectColumns($this);
            } catch (QueryIncompatibleWithFastPagination) {
                return $this->{$paginationMethod}($perPage, $columns, $pageName, $page);
            }

            $paginator = $this->clone() // @phpstan-ignore-line
                ->select($innerSelectColumns)
                ->setEagerLoads([])
                ->{$paginationMethod}($perPage, ['*'], $pageName, $page);

            $ids = $paginator->getCollection()->map->getRawOriginal($key)->toArray();

            if (in_array($model->getKeyType(), ['int', 'integer'])) {
                $this->query->whereIntegerInRaw("$table.$key", $ids); // @phpstan-ignore-line
            } else {
                $this->query->whereIn("$table.$key", $ids); // @phpstan-ignore-line
            }
            $items = $this->simplePaginate($perPage, $columns, $pageName, 1)->items();

            return Closure::fromCallable($paginatorOutput)->call($this, $items, $paginator);
        };
    }

    public static function getInnerSelectColumns(Builder $builder): array
    {
        $base = $builder->getQuery();
        $model = $builder->newModelInstance();
        $key = $model->getKeyName(); // @phpstan-ignore-line
        $table = $model->getTable(); // @phpstan-ignore-line
        $orders = collect($base->orders)
            ->pluck('column')
            ->map(function ($column) use ($base) {
                return $base->grammar->wrap($column);
            });

        return collect($base->columns)
            ->filter(function ($column) use ($orders, $base) {
                $column = $column instanceof Expression ? $column->getValue() : $base->grammar->wrap($column);
                foreach ($orders as $order) {
                    if (str_contains($column, "as $order")) {
                        return true;
                    }
                }

                // Otherwise we don't.
                return false;
            })
            ->each(/**
             * @throws QueryIncompatibleWithFastPagination
             */ function ($column) {
                if (str_contains($column, '?')) {
                    throw new QueryIncompatibleWithFastPagination();
                }
            })
            ->prepend("$table.$key")
            ->unique()
            ->values()
            ->toArray();
    }
}
