<?php

namespace Inside\Console;

use Closure;
use Exception;
use Illuminate\Console\Application as Artisan;
use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Contracts\Debug\ExceptionHandler;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Application;
use Inside\Console\Commands\InstallCommand;
use Inside\Console\Commands\RefreshCommand;
use Inside\Console\Commands\UpdateCodeCommand;
use Inside\Console\Commands\UpdateCommand;
use Inside\Facades\Package;
use Inside\Providers\ServiceProvider;
use Inside\Support\SetRequestForConsole;
use Laravel\Lumen\Console\Kernel as ConsoleKernel;
use Laravel\Lumen\Http\Request;
use ReflectionClass;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Finder\Finder;

class Kernel extends ConsoleKernel
{
    /**
     * The Artisan commands provided by your application.
     *
     * @var array
     */
    protected $commands = [
        InstallCommand::class,
        RefreshCommand::class,
        UpdateCommand::class,
        UpdateCodeCommand::class,
    ];

    protected bool $commandsLoaded = false;

    /**
     * @var Application;
     */
    protected $app;

    /** @var string[] */
    protected array $bootstrappers = [
        SetRequestForConsole::class,
    ];

    public function __construct(Application $app)
    {
        parent::__construct($app);
    }

    /**
     * Define the application's command schedule.
     * @throws Exception
     */
    protected function schedule(Schedule $schedule): void
    {
        $this->bootstrap();

        foreach (config('scheduler.schedule', []) as $callback) {
            if ($callback) {
                try {
                    $callback($schedule);
                } catch (Exception $e) {
                    Log::error('Failed to schedule task ['.$e->getMessage().']');
                }
            }
        }
    }

    public function output(): string
    {
        $this->bootstrap();

        return $this->getArtisan()->output();
    }

    /**
     * @throws Exception
     */
    public function bootstrap(): void
    {
        $this->app->boot();
        if (! $this->app->hasBeenBootstrapped()) {
            $this->app->bootstrapWith($this->bootstrappers());
        }

        if (! $this->commandsLoaded) {
            $this->commands();

            $this->commandsLoaded = true;
        }
    }

    protected function commands(): void
    {
        $this->load(__DIR__.'/Commands');
        require inside_core_path('routes/console.php');
    }

    /**
     * Register a closure based command ( via console.php routes )
     */
    public function command(string $signature, Closure $callback): ClosureCommand
    {
        $command = new ClosureCommand($signature, $callback);

        Artisan::starting(
            function ($artisan) use ($command) {
                $artisan->add($command);
            }
        );

        return $command;
    }

    /**
     * @param  InputInterface  $input
     * @param  OutputInterface|null  $output
     * @return int
     */
    public function handle($input, $output = null): int
    {
        try {
            $this->bootstrap();

            return $this->getArtisan()->run($input, $output);
        } catch (Exception $e) {
            $this->reportException($e);

            $this->renderException($output, $e);

            return 1;
        }
    }

    protected function reportException(Exception $e): void
    {
        $this->app[ExceptionHandler::class]->report($e);
    }

    /**
     * @param  OutputInterface|null  $output
     * @param  Exception  $e
     * @return void
     */
    protected function renderException($output, Exception $e): void
    {
        $this->app[ExceptionHandler::class]->renderForConsole($output, $e);
    }

    /**
     * @param  string  $command
     * @param  array  $parameters
     * @param  null  $outputBuffer
     * @return int
     *
     * @throws Exception
     */
    public function call($command, array $parameters = [], $outputBuffer = null): int
    {
        $this->bootstrap();

        if (! $this->app->runningInConsole()) {
            $this->loadProviderCommands();
        }

        return parent::call($command, $parameters, $outputBuffer);
    }

    /**
     * @throws Exception
     */
    public function all(): array
    {
        $this->bootstrap();

        return parent::all();
    }

    /**
     * @return string[]
     */
    protected function bootstrappers(): array
    {
        return $this->bootstrappers;
    }

    protected function setRequestForConsole(\Laravel\Lumen\Application $app)
    {
        $uri = $app->make('config')->get('app.url', 'http://localhost');

        $components = parse_url($uri);

        $server = $_SERVER;

        if ($components && isset($components['path'])) {
            $server = array_merge($server, [
                'SCRIPT_FILENAME' => $components['path'],
                'SCRIPT_NAME' => $components['path'],
            ]);
        }

        $app->instance('request', Request::create(
            $uri,
            'GET',
            [],
            [],
            [],
            $server
        ));
    }

    protected function load(array|string $paths): void
    {
        $paths = array_unique(Arr::wrap($paths));

        $paths = array_filter($paths, function ($path) {
            return is_dir($path);
        });

        if (empty($paths)) {
            return;
        }

        $namespace = $this->app->getNamespace();

        foreach ((new Finder())->in($paths)->files() as $command) {
            $command = $namespace.str_replace(
                ['/', '.php'],
                ['\\', ''],
                Str::after(
                    $command->getRealPath(),
                    realpath(inside_core_path()).DIRECTORY_SEPARATOR.'app'.DIRECTORY_SEPARATOR
                )
            );

            if (is_subclass_of($command, Command::class) && ! (new ReflectionClass($command))->isAbstract()) {
                Artisan::starting(function ($artisan) use ($command) {
                    $artisan->resolve($command);
                });
            }
        }
    }

    protected function loadProviderCommands(): void
    {
        foreach (Package::list() as $package) {
            foreach ($package->getProviders() as $provider) {
                $provider = $this->app->getProvider($provider);
                if ($provider instanceof ServiceProvider) {
                    $provider->registerCommands();
                }
            }
        }
    }

    public function buildSchedule(Schedule $schedule)
    {
        $this->schedule($schedule);
    }
}
